<?php
// Seed sample error log entries for preparation QA reports.
declare(strict_types=1);

$ROOT = dirname(__DIR__);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';
require_once $ROOT . '/app/Support/SoPreparacionErrors.php';

$force = in_array('--force', $argv, true);

$pdo = get_pdo();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
$pdo->exec('SET NAMES utf8mb4');

$tableName = so_prep_err_table_name();
if (!so_prep_err_table_exists($pdo, $tableName)) {
    throw new RuntimeException("Tabla {$tableName} no encontrada. Ejecute la migración antes de sembrar datos.");
}

if (!$force) {
    $alreadySeeded = (int) $pdo->query("SELECT COUNT(*) FROM {$tableName} WHERE observacion LIKE 'SEED QA %'")->fetchColumn();
    if ($alreadySeeded > 0) {
        echo json_encode([
            'status' => 'skipped',
            'reason' => 'Ya existen registros SEED QA (use --force para resembrar)',
            'existing_rows' => $alreadySeeded,
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), PHP_EOL;
        exit(0);
    }
}

$motivos = so_prep_err_fetch_motivos($pdo);
if (!$motivos) {
    throw new RuntimeException('No hay motivos cargados en para_motivos_errores.');
}

$findMotivo = static function (array $list, string $needle): ?array {
    $needleNorm = mb_strtolower(trim($needle));
    foreach ($list as $row) {
        $nombre = mb_strtolower(trim((string) ($row['nombre'] ?? '')));
        if ($nombre === $needleNorm) {
            return $row;
        }
    }
    return null;
};

$prepPickingMotivo = $findMotivo($motivos, 'preparación picking');
if (!$prepPickingMotivo) {
    throw new RuntimeException('No se encontró el motivo "Preparación picking" en para_motivos_errores.');
}

$otherMotivos = array_values(array_filter($motivos, static function (array $m) use ($prepPickingMotivo) {
    return (int) $m['id'] !== (int) $prepPickingMotivo['id'];
}));
if (count($otherMotivos) === 0) {
    // Reutilizar motivo principal para las entradas "random" si no hay más opciones.
    $otherMotivos = [$prepPickingMotivo];
}

// Obtener pedidos recientes.
$soRows = $pdo->query('SELECT id FROM so_pedido ORDER BY id DESC LIMIT 30')->fetchAll(PDO::FETCH_COLUMN);
if (count($soRows) < 7) {
    throw new RuntimeException('Se requieren al menos 7 pedidos en so_pedido para generar las semillas.');
}

$soCandidates = array_map('intval', $soRows);
shuffle($soCandidates);
$targetSoIds = array_slice($soCandidates, 0, 7);

// Obtener usuarios disponibles (opcional).
$usuarios = [];
if (so_prep_err_table_exists($pdo, 'sys_users')) {
    $nameCol = so_prep_err_detect_column($pdo, 'sys_users', ['full_name', 'nombre', 'name']);
    $altCol = so_prep_err_detect_column($pdo, 'sys_users', ['username', 'user', 'email']);

    $cols = ['id'];
    if ($nameCol) {
        $cols[] = "`{$nameCol}` AS c1";
    }
    if ($altCol && $altCol !== $nameCol) {
        $cols[] = "`{$altCol}` AS c2";
    }

    $sqlUsuarios = 'SELECT ' . implode(', ', $cols) . ' FROM sys_users ORDER BY id LIMIT 50';
    foreach ($pdo->query($sqlUsuarios)->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $label = '';
        if (isset($row['c1'])) {
            $label = trim((string) $row['c1']);
        }
        if ($label === '' && isset($row['c2'])) {
            $label = trim((string) $row['c2']);
        }
        if ($label === '') {
            $label = 'Usuario #' . $row['id'];
        }
        $usuarios[] = [
            'id' => (int) $row['id'],
            'label' => $label,
        ];
    }
}

$observaciones = [
    'Reempaque urgente por diferencia detectada en control visual.',
    'Cajas apiladas incorrectamente, se solicitó reacomodo.',
    'Faltó validar vencimiento durante la preparación inicial.',
    'Se detectó pallet con lector óptico fuera de servicio.',
    'Producto trasladado manualmente por congestión en pasillo.',
    'Control detectó rotura en film de pallet destino.',
    'Se registró demora por espera de montacargas disponible.',
];

$now = new DateTimeImmutable('now');
$inserted = [];

$pdo->beginTransaction();
try {
    for ($i = 0; $i < 7; $i++) {
        $soId = (int) $targetSoIds[$i];
        $preInfo = so_prep_err_resolve_pre_info($pdo, $soId);
        $preId = $preInfo['id'] ?? null;

        $motivo = ($i < 5) ? $prepPickingMotivo : $otherMotivos[array_rand($otherMotivos)];

        $responsableUserId = null;
        $responsableNombre = null;
        if ($usuarios) {
            $user = $usuarios[array_rand($usuarios)];
            $responsableUserId = $user['id'];
            $responsableNombre = $user['label'];
        } else {
            $responsableNombre = 'Turno Z ' . chr(65 + $i);
        }

        $observacion = sprintf('SEED QA #%d - %s', $i + 1, $observaciones[$i % count($observaciones)]);

        $hoursBack = random_int(4, 72);
        $minutesBack = random_int(0, 59);
        $loggedAt = $now->sub(new DateInterval(sprintf('PT%dH%dM', $hoursBack, $minutesBack)));

        $payload = [
            'so_id' => $soId,
            'pre_id' => $preId,
            'motivo_id' => (int) $motivo['id'],
            'responsable_user_id' => $responsableUserId,
            'responsable_nombre' => $responsableNombre,
            'observacion' => $observacion,
            'logged_at' => $loggedAt->format('Y-m-d H:i:s'),
        ];

        $id = so_prep_err_insert($pdo, $payload);
        $inserted[] = [
            'id' => $id,
            'so_id' => $soId,
            'pre_id' => $preId,
            'motivo' => $motivo['nombre'],
            'logged_at' => $payload['logged_at'],
        ];
    }

    $pdo->commit();
} catch (Throwable $e) {
    $pdo->rollBack();
    throw $e;
}

echo json_encode([
    'status' => 'ok',
    'inserted_count' => count($inserted),
    'rows' => $inserted,
    'notes' => [
        'Se generan 5 registros con motivo "Preparación picking" y 2 con motivos aleatorios.',
        'Los asientos utilizan observaciones prefijadas "SEED QA" para facilitar la depuración.',
        'Use --force para reinsertar tras limpiar la tabla o los registros seed.',
    ],
], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), PHP_EOL;
