<?php
declare(strict_types=1);

/**
 * Cambia el estado de embarques existentes utilizando la misma lógica básica
 * del endpoint api/operaciones/so_embarque_set_estado.php, permitiendo ejecutar
 * simulaciones desde CLI.
 *
 * Uso:
 *   php scripts/set_embarque_estado.php --codigo=EMB-202511-0001
 *   php scripts/set_embarque_estado.php --like=EMB-202511-% --estado=ENTREGADO
 *   php scripts/set_embarque_estado.php --like=EMB-202511-% --dry-run
 */

$ROOT = dirname(__DIR__);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';
require_once $ROOT . '/app/Support/ApiHelpers.php';

$options = getopt('', [
    'codigo::',
    'like::',
    'estado::',
    'dry-run',
    'force',
]);

$codigo = isset($options['codigo']) ? trim((string) $options['codigo']) : '';
$likePattern = isset($options['like']) ? trim((string) $options['like']) : '';
$estadoCode = strtoupper(trim((string) ($options['estado'] ?? 'ENTREGADO')));
$dryRun = array_key_exists('dry-run', $options);
$force = array_key_exists('force', $options);

if ($estadoCode === '') {
    fwrite(STDERR, "Debe indicar un estado válido mediante --estado.\n");
    exit(1);
}

if ($codigo === '' && $likePattern === '') {
    fwrite(STDERR, "Debe indicar --codigo=EMB-XXXX o --like=EMB-XXXX-% para seleccionar embarques.\n");
    exit(1);
}

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec('SET NAMES utf8mb4');

    $estadoId = fetchEstadoId($pdo, $estadoCode);
    if ($estadoId <= 0) {
        fwrite(STDERR, "Estado {$estadoCode} no encontrado en so_embarque_estado.\n");
        exit(1);
    }

    $rows = fetchEmbarques($pdo, $codigo, $likePattern);
    if (!$rows) {
        echo "No se encontraron embarques para cambiar a {$estadoCode}.\n";
        exit(0);
    }

    $hasOcupadoColumn = columnExists($pdo, 'para_moviles', 'ocupado');

    $updated = 0;
    $skipped = 0;
    foreach ($rows as $row) {
        $currentCode = strtoupper((string) $row['estado_code']);
        $codigoEmbarque = (string) $row['codigo'];

        if ($currentCode === 'FINALIZADO' && !$force) {
            echo "Omitido {$codigoEmbarque}: estado FINALIZADO no se modifica (usar --force para forzar).\n";
            $skipped++;
            continue;
        }

        if ($currentCode === $estadoCode) {
            echo "Sin cambios {$codigoEmbarque}: ya está en {$estadoCode}.\n";
            $skipped++;
            continue;
        }

        if ($dryRun) {
            echo "[dry-run] Cambiaría {$codigoEmbarque} de {$currentCode} a {$estadoCode}.\n";
            $updated++;
            continue;
        }

        $upd = $pdo->prepare('UPDATE so_embarque SET estado_id = ?, updated_at = NOW() WHERE id = ?');
        $upd->execute([$estadoId, (int) $row['id']]);

        $movilId = $row['movil_id'] !== null ? (int) $row['movil_id'] : null;
        if ($hasOcupadoColumn && $movilId && in_array($estadoCode, ['VUELTA', 'FINALIZADO'], true)) {
            $rel = $pdo->prepare('UPDATE para_moviles SET ocupado = 0, updated_at = NOW() WHERE id = ?');
            $rel->execute([$movilId]);
        }

        echo "Actualizado {$codigoEmbarque}: {$currentCode} -> {$estadoCode}.\n";
        $updated++;
    }

    echo "\nResumen: {$updated} cambios, {$skipped} omitidos" . ($dryRun ? " (dry-run)" : '') . ".\n";
    exit(0);
} catch (Throwable $e) {
    fwrite(STDERR, "Error: {$e->getMessage()}\n");
    exit(1);
}

function fetchEmbarques(PDO $pdo, string $codigo, string $likePattern): array
{
    if ($codigo !== '') {
        $stmt = $pdo->prepare(
            'SELECT e.id, e.codigo, e.movil_id, ee.code AS estado_code
             FROM so_embarque e
             JOIN so_embarque_estado ee ON ee.id = e.estado_id
            WHERE e.codigo = ?'
        );
        $stmt->execute([$codigo]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
    }

    $stmt = $pdo->prepare(
        'SELECT e.id, e.codigo, e.movil_id, ee.code AS estado_code
           FROM so_embarque e
           JOIN so_embarque_estado ee ON ee.id = e.estado_id
          WHERE e.codigo LIKE ?
          ORDER BY e.codigo'
    );
    $stmt->execute([$likePattern]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
}

function fetchEstadoId(PDO $pdo, string $code): int
{
    $stmt = $pdo->prepare('SELECT id FROM so_embarque_estado WHERE code = ? LIMIT 1');
    $stmt->execute([$code]);
    return (int) $stmt->fetchColumn();
}
