<?php
// scripts/sp_preparar_replay.php
// Reproduce la lógica de sp_so_preparar_auto SIN usar el SP.
// Ejecuta query por query, imprime tiempos, cantidad de filas y arma un plan en modo simulación.
// Uso (PowerShell):
//   php scripts/sp_preparar_replay.php --id=3 --dep=DEP1 --simulate --direct --cands=200 --json=tmp_plan.json
//   php scripts/sp_preparar_replay.php --code=SO-... --dep=DEP1 --simulate --cands=100 --verbose

declare(strict_types=1);

require_once __DIR__ . '/../config/db.php';

function opts(): array {
    static $o = null; if ($o!==null) return $o;
    $o = getopt('', [
        'id::','code::','dep::','pos::',
        'simulate','direct','cands::','verbose','json::','timeout::'
    ]);
    return $o;
}

function hasOpt(string $k): bool { $o = opts(); return array_key_exists($k, $o); }
function opt(string $k, $def=null) { $o = opts(); return array_key_exists($k, $o) ? $o[$k] : $def; }

function out(string $s=""): void { echo $s, PHP_EOL; }
function logq(string $label, string $sql, float $sec, int $rows): void {
    out(sprintf("[%s] %.3fs | filas=%d | %s", $label, $sec, $rows, preg_replace('/\s+/', ' ', trim($sql))));
}

function runRows(PDO $pdo, string $label, string $sql, array $params=[]): array {
    $t0 = microtime(true);
    $st = $pdo->prepare($sql);
    $st->execute($params);
    $rows = $st->fetchAll(PDO::FETCH_ASSOC);
    $sec = microtime(true)-$t0;
    logq($label, $sql, $sec, count($rows));
    return $rows;
}

function runOne(PDO $pdo, string $label, string $sql, array $params=[]) {
    $rows = runRows($pdo, $label, $sql, $params); return $rows[0] ?? null;
}

function execQ(PDO $pdo, string $label, string $sql, array $params=[]): int {
    $t0 = microtime(true);
    $st = $pdo->prepare($sql);
    $ok = $st->execute($params);
    $sec = microtime(true)-$t0;
    logq($label, $sql, $sec, $ok ? $st->rowCount() : -1);
    return $st->rowCount();
}

function main(): int {
    $soId   = opt('id');
    $soCode = opt('code');
    $dep    = opt('dep','DEP1');
    $posInp = opt('pos');
    $simulate = hasOpt('simulate');
    $direct   = hasOpt('direct');
    $cands    = opt('cands'); $cands = $cands!==null ? max(1,(int)$cands) : null;
    $verbose  = hasOpt('verbose');
    $jsonOut  = opt('json');
    $timeout  = (int)opt('timeout', 60);

    $pdo = getPDO();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec("SET NAMES utf8mb4 COLLATE utf8mb4_0900_ai_ci");
    $pdo->exec("SET collation_connection = 'utf8mb4_0900_ai_ci'");
    $pdo->exec('SET SESSION innodb_lock_wait_timeout = ' . $timeout);
    $pdo->exec('SET SESSION lock_wait_timeout = ' . $timeout);
    $pdo->exec('SET SESSION transaction_isolation = "READ-COMMITTED"');
    $pdo->exec('SET SESSION autocommit = 1');

    if (!$soCode) {
        if (!$soId) { out('Debe indicar --id o --code'); return 2; }
        $r = runOne($pdo, 'GET SO CODE', 'SELECT codigo FROM so_pedido WHERE id=? LIMIT 1', [(int)$soId]);
        if (!$r) { out('Pedido no encontrado'); return 1; }
        $soCode = (string)$r['codigo'];
    }

    out("Pedido: $soCode | Depósito: $dep | Simulate=".($simulate?1:0)." | Direct=".($direct?1:0)." | Cands=".($cands??'ALL'));

    // 1) IDs básicos
    $r = runOne($pdo, 'GET SO ID', 'SELECT id FROM so_pedido WHERE codigo=? LIMIT 1', [$soCode]);
    if (!$r) { out('Pedido no encontrado'); return 1; }
    $pedidoId = (int)$r['id'];

    $r = runOne($pdo, 'GET DEP ID', 'SELECT id FROM wh_deposito WHERE code=? LIMIT 1', [$dep]);
    if (!$r) { out('Depósito no encontrado'); return 1; }
    $depId = (int)$r['id'];

    // 2) Pos PREP
    if ($posInp) {
        $r = runOne($pdo, 'GET PREP POS', "SELECT p.id FROM wh_posicion p WHERE p.deposito_id=? AND p.activo=1 AND (p.code=? OR p.code_full=? OR p.pos_code=? OR p.pos_code_full=?) LIMIT 1", [$depId,$posInp,$posInp,$posInp,$posInp]);
        if (!$r) { out('Posición PREP indicada no existe'); return 1; }
        $posPrepId = (int)$r['id'];
    } else {
        $r = runOne($pdo, 'FIND PREP POS', "SELECT p.id FROM wh_posicion p JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PREP' LEFT JOIN wh_stock s ON s.posicion_id=p.id AND s.deposito_id=? WHERE p.deposito_id=? AND p.activo=1 GROUP BY p.id ORDER BY COALESCE(SUM(s.qty_uv+s.qty_uc),0) ASC, p.id ASC LIMIT 1", [$depId,$depId]);
        if (!$r) { out('No hay posiciones PREP'); return 1; }
        $posPrepId = (int)$r['id'];
    }

    // 3) Pos PICKING
    $r = runOne($pdo, 'FIND PICK POS', "SELECT p.id FROM wh_posicion p JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PICKING' LEFT JOIN wh_stock s ON s.posicion_id=p.id AND s.deposito_id=? WHERE p.deposito_id=? AND p.activo=1 GROUP BY p.id ORDER BY COALESCE(SUM(s.qty_uv+s.qty_uc),0) ASC, p.id ASC LIMIT 1", [$depId,$depId]);
    if (!$r) { out('No hay posiciones PICKING'); return 1; }
    $posPickId = (int)$r['id'];

    // 4) Pre-embarque
    $preCode = 'PRE-' . $soCode;
    $r = runOne($pdo, 'GET PRE ESTADO', "SELECT id FROM so_preembarque_estado WHERE code='PENDIENTE' LIMIT 1");
    if (!$r) { out('Falta tabla/estado so_preembarque_estado'); return 1; }
    $preEstId = (int)$r['id'];
    $r = runOne($pdo, 'GET PRE BY CODE', 'SELECT id FROM so_preembarque WHERE codigo=? LIMIT 1', [$preCode]);
    if (!$r) {
        if ($simulate) {
            // En modo simulación, el SP crea el PRE, pero para el replay podemos continuar sin persistir.
            $preId = 0;
        } else {
            execQ($pdo, 'INS PRE', 'INSERT INTO so_preembarque (codigo, pedido_id, deposito_id, estado_id, zona_posicion_id, asignado_at, inicio_at) VALUES (?,?,?,?,?,?,NOW())', [$preCode,$pedidoId,$depId,$preEstId,$posPrepId, date('Y-m-d H:i:s')]);
            $r = runOne($pdo, 'GET PRE BY CODE 2', 'SELECT id FROM so_preembarque WHERE codigo=? LIMIT 1', [$preCode]);
            if (!$r) { out('No se pudo crear/obtener pre-embarque'); return 1; }
            $preId = (int)$r['id'];
        }
    } else {
        if (!$simulate) {
            execQ($pdo, 'UPD PRE ZONA', 'UPDATE so_preembarque SET zona_posicion_id=COALESCE(zona_posicion_id, ?) WHERE id=?', [$posPrepId, (int)$r['id']]);
        }
        $preId = (int)$r['id'];
    }

    // 5) Ítems pendientes
    $pend = runRows($pdo, 'PEND ITEMS', 'SELECT i.id AS pedido_dest_item_id, d.id AS pedido_dest_id, d.destinatario_id, i.producto_id, i.lote_codigo, GREATEST(i.expected_uv - i.prepared_uv,0) AS need_uv, GREATEST(i.expected_uc - i.prepared_uc,0) AS need_uc FROM so_pedido_dest_item i JOIN so_pedido_dest d ON d.id=i.pedido_dest_id WHERE d.pedido_id=? AND (GREATEST(i.expected_uv - i.prepared_uv,0) > 0 OR GREATEST(i.expected_uc - i.prepared_uc,0) > 0)', [$pedidoId]);

    $plan = [];
    foreach ($pend as $it) {
        $itemId = (int)$it['pedido_dest_item_id'];
        $prodId = (int)$it['producto_id'];
        $loteCode = $it['lote_codigo'] !== null ? (string)$it['lote_codigo'] : null;
        $needUv = (int)$it['need_uv'];
        $needUc = (int)$it['need_uc'];
        out(sprintf('ITEM %d prod=%d lote=%s needs UV=%d UC=%d', $itemId, $prodId, $loteCode??'NULL', $needUv, $needUc));

        // 5.a Candidatos en PICKING por FEFO
        $pickSql = "SELECT s.posicion_id, s.lote_id, s.pallet_id, s.qty_uv, s.qty_uc, l.fecha_vencimiento FROM wh_stock s JOIN wh_posicion p ON p.id=s.posicion_id JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PICKING' LEFT JOIN wh_lote l ON l.id=s.lote_id WHERE s.deposito_id=? AND s.producto_id=? AND ( ? IS NULL OR ?='' OR EXISTS(SELECT 1 FROM wh_lote lx WHERE lx.id=s.lote_id AND lx.codigo=?)) AND (s.qty_uv>0 OR s.qty_uc>0) ORDER BY ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC";
        if ($cands) { $pickSql .= ' LIMIT '.(int)$cands; }
        $candsPick = runRows($pdo, 'CAND PICK', $pickSql, [$depId,$prodId,$loteCode,$loteCode,$loteCode]);

        // Consumir desde PICKING
        while (($needUv>0 || $needUc>0) && !empty($candsPick)) {
            $cand = array_shift($candsPick);
            $posFrom = (int)$cand['posicion_id'];
            $loteId  = $cand['lote_id']!==null ? (int)$cand['lote_id'] : null;
            $palletId= $cand['pallet_id']!==null ? (int)$cand['pallet_id'] : null;

            // Revalidar cantidades actuales en wh_stock
            $reval = runOne($pdo, 'REVAL STOCK', 'SELECT qty_uv, qty_uc FROM wh_stock WHERE deposito_id=? AND posicion_id=? AND producto_id=? AND lote_id <=> ? AND pallet_id <=> ? LIMIT 1', [$depId,$posFrom,$prodId,$loteId,$palletId]);
            $curUv = (int)($reval['qty_uv'] ?? 0); $curUc = (int)($reval['qty_uc'] ?? 0);
            $takeUv = min($curUv, $needUv); $takeUc = min($curUc, $needUc);
            if ($takeUv===0 && $takeUc===0) continue;

            if ($simulate) {
                $plan[] = ['fase'=>'PICK','item_id'=>$itemId,'producto_id'=>$prodId,'lote_id'=>$loteId,'lote_code'=>$loteCode,'from_pos_id'=>$posFrom,'to_pos_id'=>$posPrepId,'pallet_id'=>$palletId,'take_uv'=>$takeUv,'take_uc'=>$takeUc,'venc'=>null];
            } else {
                execQ($pdo,'INS MOVE PICK','INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia) VALUES (?,?,?,?,?,?,?,?,?,?,?)', [$depId,'MOVE','PREPARACION',$palletId,$prodId,$loteId,$posFrom,$posPrepId,$takeUv,$takeUc,'PRE-'.$pedidoId]);
                execQ($pdo,'INS SO_PRE_PICK','INSERT INTO so_pre_pick (preembarque_id, pedido_dest_item_id, from_pos_id, to_pos_id, pallet_id, lote_id, uv_cajas, uc_unidades, creado_por) VALUES (?,?,?,?,?,?,?,?,NULL)', [$preId,$itemId,$posFrom,$posPrepId,$palletId,$loteId,$takeUv,$takeUc]);
            }
            $needUv -= $takeUv; $needUc -= $takeUc;
        }

        // 5.b Reposición si falta
        while ($needUv>0 || $needUc>0) {
            $srcSql = "SELECT s.posicion_id AS from_pos_id, s.pallet_id, s.lote_id, s.qty_uv, s.qty_uc, l.fecha_vencimiento FROM wh_stock s JOIN wh_posicion p ON p.id=s.posicion_id JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code NOT IN ('PREP','CUARENTENA','PICKING') LEFT JOIN wh_lote l ON l.id=s.lote_id WHERE s.deposito_id=? AND s.producto_id=? AND ( ? IS NULL OR ?='' OR EXISTS(SELECT 1 FROM wh_lote lx WHERE lx.id=s.lote_id AND lx.codigo=?)) AND (s.qty_uv>0 OR s.qty_uc>0) ORDER BY ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC";
            if ($cands) { $srcSql .= ' LIMIT '.(int)$cands; }
            $rows = runRows($pdo, 'CAND REPO', $srcSql, [$depId,$prodId,$loteCode,$loteCode,$loteCode]);
            if (empty($rows)) { out('  -> Sin más candidatos para reponer'); break; }
            $best = $rows[0];
            $posFrom = (int)$best['from_pos_id'];
            $loteId  = $best['lote_id']!==null ? (int)$best['lote_id'] : null;
            $palletId= $best['pallet_id']!==null ? (int)$best['pallet_id'] : null;
            $takeUv  = min((int)($best['qty_uv'] ?? 0), $needUv);
            $takeUc  = min((int)($best['qty_uc'] ?? 0), $needUc);
            if ($takeUv===0 && $takeUc===0) break;

            if ($direct) {
                if ($simulate) {
                    $plan[] = ['fase'=>'DIRECT','item_id'=>$itemId,'producto_id'=>$prodId,'lote_id'=>$loteId,'lote_code'=>$loteCode,'from_pos_id'=>$posFrom,'to_pos_id'=>$posPrepId,'pallet_id'=>$palletId,'take_uv'=>$takeUv,'take_uc'=>$takeUc,'venc'=>null];
                } else {
                    execQ($pdo,'INS MOVE DIRECT','INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia) VALUES (?,?,?,?,?,?,?,?,?,?,?)', [$depId,'MOVE','PREPARACION',$palletId,$prodId,$loteId,$posFrom,$posPrepId,$takeUv,$takeUc,'PRE-'.$pedidoId]);
                    execQ($pdo,'INS SO_PRE_PICK DIR','INSERT INTO so_pre_pick (preembarque_id, pedido_dest_item_id, from_pos_id, to_pos_id, pallet_id, lote_id, uv_cajas, uc_unidades, creado_por) VALUES (?,?,?,?,?,?,?,?,NULL)', [$preId,$itemId,$posFrom,$posPrepId,$palletId,$loteId,$takeUv,$takeUc]);
                }
                $needUv -= $takeUv; $needUc -= $takeUc;
            } else {
                // Reponer a PICKING y luego PICK2
                if ($simulate) {
                    $plan[] = ['fase'=>'REPO','item_id'=>$itemId,'producto_id'=>$prodId,'lote_id'=>$loteId,'lote_code'=>$loteCode,'from_pos_id'=>$posFrom,'to_pos_id'=>$posPickId,'pallet_id'=>$palletId,'take_uv'=>$takeUv,'take_uc'=>$takeUc,'venc'=>null];
                } else {
                    execQ($pdo,'INS MOVE REPO','INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia) VALUES (?,?,?,?,?,?,?,?,?,?,?)', [$depId,'MOVE','REPOSICION_PICKING',$palletId,$prodId,$loteId,$posFrom,$posPickId,$takeUv,$takeUc,'PREP-REP-'.$pedidoId]);
                }

                // Volver a buscar en PICKING
                $pick2Sql = "SELECT s.posicion_id, s.lote_id, s.pallet_id, s.qty_uv, s.qty_uc, l.fecha_vencimiento FROM wh_stock s JOIN wh_posicion p ON p.id=s.posicion_id JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PICKING' LEFT JOIN wh_lote l ON l.id=s.lote_id WHERE s.deposito_id=? AND s.producto_id=? AND ( ? IS NULL OR ?='' OR EXISTS(SELECT 1 FROM wh_lote lx WHERE lx.id=s.lote_id AND lx.codigo=?)) AND (s.qty_uv>0 OR s.qty_uc>0) ORDER BY ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC";
                if ($cands) { $pick2Sql .= ' LIMIT '.(int)$cands; }
                $candsPick2 = runRows($pdo, 'CAND PICK2', $pick2Sql, [$depId,$prodId,$loteCode,$loteCode,$loteCode]);
                while (($needUv>0 || $needUc>0) && !empty($candsPick2)) {
                    $cand = array_shift($candsPick2);
                    $posFrom2 = (int)$cand['posicion_id'];
                    $loteId2  = $cand['lote_id']!==null ? (int)$cand['lote_id'] : null;
                    $palletId2= $cand['pallet_id']!==null ? (int)$cand['pallet_id'] : null;
                    $reval2 = runOne($pdo, 'REVAL STOCK2', 'SELECT qty_uv, qty_uc FROM wh_stock WHERE deposito_id=? AND posicion_id=? AND producto_id=? AND lote_id <=> ? AND pallet_id <=> ? LIMIT 1', [$depId,$posFrom2,$prodId,$loteId2,$palletId2]);
                    $curUv2 = (int)($reval2['qty_uv'] ?? 0); $curUc2 = (int)($reval2['qty_uc'] ?? 0);
                    $takeUv2 = min($curUv2, $needUv); $takeUc2 = min($curUc2, $needUc);
                    if ($takeUv2===0 && $takeUc2===0) continue;
                    if ($simulate) {
                        $plan[] = ['fase'=>'PICK2','item_id'=>$itemId,'producto_id'=>$prodId,'lote_id'=>$loteId2,'lote_code'=>$loteCode,'from_pos_id'=>$posFrom2,'to_pos_id'=>$posPrepId,'pallet_id'=>$palletId2,'take_uv'=>$takeUv2,'take_uc'=>$takeUc2,'venc'=>null];
                    } else {
                        execQ($pdo,'INS MOVE PICK2','INSERT INTO wh_move (deposito_id, tipo, motivo, pallet_id, producto_id, lote_id, from_pos_id, to_pos_id, delta_uv, delta_uc, referencia) VALUES (?,?,?,?,?,?,?,?,?,?,?)', [$depId,'MOVE','PREPARACION',$palletId2,$prodId,$loteId2,$posFrom2,$posPrepId,$takeUv2,$takeUc2,'PRE-'.$pedidoId]);
                        execQ($pdo,'INS SO_PRE_PICK2','INSERT INTO so_pre_pick (preembarque_id, pedido_dest_item_id, from_pos_id, to_pos_id, pallet_id, lote_id, uv_cajas, uc_unidades, creado_por) VALUES (?,?,?,?,?,?,?,?,NULL)', [$preId,$itemId,$posFrom2,$posPrepId,$palletId2,$loteId2,$takeUv2,$takeUc2]);
                    }
                    $needUv -= $takeUv2; $needUc -= $takeUc2;
                }
            }
            // Si ya no hay necesidades, salimos del while reponer
            if (!($needUv>0 || $needUc>0)) break;
        }

        out(sprintf('  -> Remaining UV=%d UC=%d', $needUv, $needUc));
    }

    // Resumen final (como el SP)
    $summary = runRows($pdo, 'FINAL SUMMARY', 'SELECT d.pedido_id, i.id AS pedido_dest_item_id, i.producto_id, i.lote_codigo, i.expected_uv, i.prepared_uv, (i.expected_uv - i.prepared_uv) AS pend_uv, i.expected_uc, i.prepared_uc, (i.expected_uc - i.prepared_uc) AS pend_uc FROM so_pedido_dest_item i JOIN so_pedido_dest d ON d.id=i.pedido_dest_id WHERE d.pedido_id=?', [$pedidoId]);

    if ($jsonOut) {
        $report = ['pedido'=>$soCode,'dep'=>$dep,'simulate'=>$simulate?1:0,'direct'=>$direct?1:0,'cands'=>$cands,'plan'=>$plan,'summary'=>$summary,'ts'=>date('c')];
        file_put_contents($jsonOut, json_encode($report, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
        out('Guardado JSON en '.$jsonOut);
    }

    out('Done.');
    return 0;
}

try { exit(main()); } catch (Throwable $e) { fwrite(STDERR, 'ERROR: '.$e->getMessage()."\n"); exit(1); }
