<?php
// scripts/sp_preparar_scenarios.php
// Herramienta de línea de comandos para probar sp_so_preparar_auto con distintos escenarios.
// Uso básico (Windows PowerShell):
//   php scripts/sp_preparar_scenarios.php --so-id=3 --dep=DEP1 --simulate --direct --log --timeout=120 --cands=200 --dump-sets --json=tmp_out.json
//   php scripts/sp_preparar_scenarios.php --code=SO-20251015-012435 --dep=DEP1 --compare-direct --simulate

declare(strict_types=1);

require_once __DIR__ . '/../config/db.php';

function out(string $s = ""): void { echo $s, PHP_EOL; }
function err(string $s): void { fwrite(STDERR, $s . PHP_EOL); }
function env_collation(PDO $pdo): void {
    try {
        $pdo->exec("SET NAMES utf8mb4 COLLATE utf8mb4_0900_ai_ci");
        $pdo->exec("SET collation_connection = 'utf8mb4_0900_ai_ci'");
    } catch (Throwable $e) {
        error_log('No se pudo fijar la collation de sesión: ' . $e->getMessage());
    }
}

function usage(): void {
    out('Pruebas de sp_so_preparar_auto');
    out('');
    out('Parámetros:');
    out('  --so-id=N                 ID del pedido (alternativa a --code)');
    out('  --code=SO-...             Código del pedido (si se omite --so-id)');
    out('  --dep=DEP1                Código de depósito (ej.: DEP1)');
    out('  --pos=PREP                Código de posición de preparación (opcional)');
    out('  --simulate                Ejecuta en modo simulación (no escribe)');
    out('  --no-simulate             Fuerza modo no-simulación');
    out('  --direct                  Activa camino directo a PREP (@so_pre_direct_to_prep=1)');
    out('  --no-direct               Desactiva camino directo');
    out('  --log                     Activa logging del SP (@so_pre_log=1)');
    out('  --no-log                  Desactiva logging del SP');
    out('  --cands=NUM               Límite de candidatos (@so_pre_max_cands si el SP lo soporta)');
    out('  --timeout=SEG             Lock wait/timeout en segundos (default 120)');
    out('  --repeat=N                Repite N veces (para promediar tiempos)');
    out('  --compare-direct          Corre dos ejecuciones: direct=0 y direct=1');
    out('  --dump-sets               Imprime los result sets en JSON');
    out('  --json=archivo.json       Guarda un JSON con métricas y (opcionalmente) sets');
    out('  --help                    Muestra esta ayuda y sale');
    out('');
    out('Ejemplos:');
    out('  php scripts/sp_preparar_scenarios.php --so-id=3 --dep=DEP1 --simulate --dump-sets');
    out('  php scripts/sp_preparar_scenarios.php --code=SO-20251015-012435 --dep=DEP1 --compare-direct --simulate --json=tmp_out.json');
}

$options = getopt('', [
    'so-id::', 'code::', 'dep::', 'pos::',
    'simulate', 'no-simulate',
    'direct', 'no-direct',
    'log', 'no-log',
    'cands::', 'iters::', 'timeout::', 'repeat::',
    'compare-direct', 'dump-sets', 'json::', 'help'
]);

if (isset($options['help'])) {
    usage();
    exit(0);
}

$soId     = isset($options['so-id']) ? (int)$options['so-id'] : null;
$soCode   = isset($options['code']) ? (string)$options['code'] : null;
$depCode  = isset($options['dep']) ? (string)$options['dep'] : 'DEP1';
$posCode  = $options['pos'] ?? null;
$simulate = isset($options['simulate']) ? 1 : (isset($options['no-simulate']) ? 0 : 1); // default simulate
$direct   = isset($options['direct']) ? 1 : (isset($options['no-direct']) ? 0 : 0);
$log      = isset($options['log']) ? 1 : (isset($options['no-log']) ? 0 : 0);
$cands    = isset($options['cands']) ? (int)$options['cands'] : null;
$iters    = isset($options['iters']) ? max(1, (int)$options['iters']) : null;
$timeout  = isset($options['timeout']) ? max(1, (int)$options['timeout']) : 120;
$repeat   = isset($options['repeat']) ? max(1, (int)$options['repeat']) : 1;
$compare  = isset($options['compare-direct']);
$dumpSets = isset($options['dump-sets']);
$jsonOut  = isset($options['json']) ? (string)$options['json'] : null;

try {
    $pdo = getPDO();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    env_collation($pdo);

    // Info básica
    $ver = $pdo->query('SELECT VERSION()')->fetchColumn();
    out('Conectado a MySQL ' . $ver);

    // Resolver código del pedido si sólo tenemos ID
    if (!$soCode) {
        if (!$soId) {
            err('Debe indicar --so-id o --code');
            usage();
            exit(2);
        }
        $st = $pdo->prepare('SELECT codigo FROM so_pedido WHERE id=? LIMIT 1');
        $st->execute([$soId]);
        $soCode = (string)$st->fetchColumn();
        if (!$soCode) {
            err('No se encontró el pedido con id=' . $soId);
            exit(1);
        }
    }

    // Verificar SP
    $chk = $pdo->query("SHOW PROCEDURE STATUS WHERE Name='sp_so_preparar_auto'")->fetch(PDO::FETCH_ASSOC);
    if (!$chk) {
        err('❌ No existe sp_so_preparar_auto. Aplique migraciones/instalación.');
        exit(1);
    }

    // Configuración de sesión
    $pdo->exec('SET SESSION innodb_lock_wait_timeout = ' . $timeout);
    $pdo->exec('SET SESSION lock_wait_timeout = ' . $timeout);
    $pdo->exec('SET SESSION transaction_isolation = "READ-COMMITTED"');
    $pdo->exec('SET SESSION autocommit = 1');

    // Preparar función de ejecución
    $run = function (int $simulateFlag, int $directFlag, int $logFlag) use ($pdo, $soCode, $depCode, $posCode, $dumpSets): array {
        // Variables de sesión controlando SP
        $pdo->exec('SET @so_pre_log := ' . $logFlag);
        $pdo->exec('SET @so_pre_direct_to_prep := ' . $directFlag);
        // Nota: @so_pre_max_cands es opcional; sólo se aplica si el SP lo soporta
        // Se establece fuera de este closure por simplicidad

        $metrics = [
            'ok' => false,
            'error' => null,
            'seconds' => 0.0,
            'resultSets' => 0,
            'rowsPerSet' => [],
        ];
        $allSets = [];

        $t0 = microtime(true);
        try {
            $stmt = $pdo->prepare('CALL sp_so_preparar_auto(?,?,?,?)');
            $stmt->execute([$soCode, $depCode, $posCode, $simulateFlag]);
            $i = 0;
            do {
                $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
                if ($rows !== false && !empty($rows)) {
                    $i++;
                    $metrics['rowsPerSet'][] = count($rows);
                    if ($dumpSets) {
                        $allSets[] = $rows;
                    }
                }
            } while ($stmt->nextRowset());
            $stmt->closeCursor();

            $metrics['resultSets'] = $i;
            $metrics['ok'] = true;
        } catch (Throwable $e) {
            $metrics['error'] = $e->getMessage();
        } finally {
            $metrics['seconds'] = round(microtime(true) - $t0, 3);
        }

        return ['metrics' => $metrics, 'sets' => $allSets];
    };

    if ($cands !== null) {
        // Establecer límite de candidatos si el SP lo aprovecha
        try { $pdo->exec('SET @so_pre_max_cands := ' . (int)$cands); } catch (Throwable $e) { /* opcional */ }
    }
    if ($iters !== null) {
        try { $pdo->exec('SET @so_pre_max_iters := ' . (int)$iters); } catch (Throwable $e) { /* opcional */ }
    }

    $scenarios = [];
    if ($compare) {
        $scenarios[] = ['label' => 'direct=0', 'direct' => 0, 'simulate' => $simulate, 'log' => $log];
        $scenarios[] = ['label' => 'direct=1', 'direct' => 1, 'simulate' => $simulate, 'log' => $log];
    } else {
        $scenarios[] = ['label' => 'custom', 'direct' => $direct, 'simulate' => $simulate, 'log' => $log];
    }

    $report = [
        'pedido' => $soCode,
        'deposito' => $depCode,
        'pos_preparacion' => $posCode,
        'repeticiones' => $repeat,
        'timeout' => $timeout,
        'candidatos' => $cands,
        'resultados' => [],
        'timestamp' => date('c'),
    ];

    foreach ($scenarios as $sc) {
        out(str_repeat('-', 60));
        out(sprintf('Escenario: %s | simulate=%d | direct=%d | log=%d', $sc['label'], $sc['simulate'], $sc['direct'], $sc['log']));

        $aggSeconds = 0.0; $aggOk = 0; $aggRows = [];
        $lastSets = [];
        for ($r = 1; $r <= $repeat; $r++) {
            if ($cands !== null) {
                try { $pdo->exec('SET @so_pre_max_cands := ' . (int)$cands); } catch (Throwable $e) { /* opcional */ }
            }
            if ($iters !== null) {
                try { $pdo->exec('SET @so_pre_max_iters := ' . (int)$iters); } catch (Throwable $e) { /* opcional */ }
            }
            $res = $run((int)$sc['simulate'], (int)$sc['direct'], (int)$sc['log']);
            $m = $res['metrics'];
            $aggSeconds += (float)$m['seconds'];
            $aggOk += $m['ok'] ? 1 : 0;
            $aggRows[] = $m['rowsPerSet'];
            $lastSets = $res['sets'];
            out(sprintf('  Ejec %d/%d: %s en %.3f s, resultSets=%d %s', $r, $repeat, $m['ok'] ? 'OK' : 'ERROR', $m['seconds'], $m['resultSets'], $m['error'] ? ('-> ' . $m['error']) : ''));
        }
        $avg = round($aggSeconds / max(1, $repeat), 3);
        out(sprintf('  Promedio: %.3f s | OK=%d/%d', $avg, $aggOk, $repeat));

        $item = [
            'label' => $sc['label'],
            'simulate' => (int)$sc['simulate'],
            'direct' => (int)$sc['direct'],
            'log' => (int)$sc['log'],
            'avg_seconds' => $avg,
            'ok_runs' => $aggOk,
            'runs' => $repeat,
            'rows_per_run' => $aggRows,
        ];
        if ($dumpSets) { $item['last_sets'] = $lastSets; }
        $report['resultados'][] = $item;
    }

    if ($jsonOut) {
        file_put_contents($jsonOut, json_encode($report, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        out('Guardado reporte JSON en ' . $jsonOut);
    }

    out(str_repeat('-', 60));
    out('Listo.');
    exit(0);
} catch (Throwable $e) {
    err('ERROR: ' . $e->getMessage());
    exit(1);
}
