<?php
declare(strict_types=1);

/**
 * Carga datos de salidas desde un CSV y los inserta en staging (so_import_batch / so_import_row).
 *
 * Ejemplos:
 *   php scripts/stage_salidas_from_csv.php
 *   php scripts/stage_salidas_from_csv.php --dry-run --limit=3
 *   php scripts/stage_salidas_from_csv.php --batch-prefix=AUTO --order-prefix=SO-BATCH --offset=10 --limit=20
 *
 * Una vez creado el batch, se puede confirmar con:
 *   php scripts/confirm_salidas_batch.php --batch-id=123 --cliente-id=1
 */

$ROOT = dirname(__DIR__);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

$options = getopt('', [
    'csv::',
    'cliente-id::',
    'batch-prefix::',
    'order-prefix::',
    'offset::',
    'limit::',
    'dry-run'
]);

$csvPath     = $options['csv'] ?? ($ROOT . '/tests/data/salidas_noviembre.csv');
$clienteId   = isset($options['cliente-id']) ? (int)$options['cliente-id'] : 1;
$batchPrefix = $options['batch-prefix'] ?? 'AUTO';
$orderPrefix = $options['order-prefix'] ?? 'SO-BATCH';
$offset      = isset($options['offset']) ? max(0, (int)$options['offset']) : 0;
$limit       = isset($options['limit']) ? max(0, (int)$options['limit']) : 0;
$dryRun      = array_key_exists('dry-run', $options);

if (!is_file($csvPath)) {
    fwrite(STDERR, "CSV not found: {$csvPath}\n");
    exit(1);
}

$rawRows = loadCsv($csvPath);
if ($offset > 0) {
    $rawRows = array_slice($rawRows, $offset);
}
if ($limit > 0) {
    $rawRows = array_slice($rawRows, 0, $limit);
}
if (!$rawRows) {
    fwrite(STDERR, "No rows to process after applying offset/limit.\n");
    exit(0);
}

$stagingRows = [];
$warnings = [];
$orderSeq = $offset;
foreach ($rawRows as $rowIdx => $row) {
    $orderSeq++;
    $items = buildItemsFromRow($row);
    if (!$items) {
        $warnings[] = sprintf('Row %d skipped: no SKU quantities found.', $offset + $rowIdx + 1);
        continue;
    }

    $orderCode = sprintf('%s-%04d', $orderPrefix, $orderSeq);
    $batchCode = sprintf('%s-%04d', $batchPrefix, $orderSeq);
    $destPayload = buildDestinatarioPayload($row, $orderCode);
    $docPayload = buildDocumentoPayload($row, $orderCode);
    $fechaSalida = $row['fecha_programada'] ?? date('Y-m-d');

    foreach ($items as $item) {
        $stagingRows[] = buildStagingRow($destPayload, $docPayload, $fechaSalida, $batchCode, $item);
    }
}

if (!$stagingRows) {
    fwrite(STDERR, "No staging rows generated.\n");
    foreach ($warnings as $warn) {
        fwrite(STDERR, " - {$warn}\n");
    }
    exit(1);
}

if ($dryRun) {
    echo "[DRY-RUN] Would create batch with " . count($stagingRows) . " staging rows for cliente {$clienteId}.\n";
    echo "Sample rows:\n";
    foreach (array_slice($stagingRows, 0, 5) as $sample) {
        echo json_encode($sample, JSON_UNESCAPED_UNICODE), "\n";
    }
    if ($warnings) {
        echo "Warnings:\n";
        foreach ($warnings as $warn) {
            echo ' - ' . $warn . "\n";
        }
    }
    exit(0);
}

$pdo = get_pdo();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
$pdo->exec('SET NAMES utf8mb4');

$pdo->beginTransaction();
try {
    $insertBatch = $pdo->prepare('INSERT INTO so_import_batch (tipo, filename, rows_total, rows_ok, rows_error) VALUES (?,?,?,?,?)');
    $insertBatch->execute(['PEDIDO_SALIDA', basename($csvPath), 0, 0, 0]);
    $batchId = (int)$pdo->lastInsertId();

    $insertRow = $pdo->prepare('INSERT INTO so_import_row (batch_id, rownum, raw) VALUES (?,?,?)');
    foreach ($stagingRows as $idx => $payload) {
        $json = json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        if ($json === false) {
            throw new RuntimeException('JSON encode failed: ' . json_last_error_msg());
        }
        $insertRow->execute([$batchId, $idx + 1, $json]);
    }

    $updateBatch = $pdo->prepare('UPDATE so_import_batch SET rows_total=?, rows_ok=?, rows_error=? WHERE id=?');
    $updateBatch->execute([count($stagingRows), 0, 0, $batchId]);

    $pdo->commit();
} catch (Throwable $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    fwrite(STDERR, 'Failed to stage rows: ' . $e->getMessage() . "\n");
    exit(1);
}

echo "Batch created: {$batchId}\n";
echo 'Staging rows inserted: ' . count($stagingRows) . "\n";
echo 'Use: php scripts/confirm_salidas_batch.php --batch-id=' . $batchId . ' --cliente-id=' . $clienteId . "\n";
if ($warnings) {
    echo "Warnings:\n";
    foreach ($warnings as $warn) {
        echo ' - ' . $warn . "\n";
    }
}

function loadCsv(string $path): array
{
    $fp = fopen($path, 'r');
    if ($fp === false) {
        throw new RuntimeException('Could not open CSV: ' . $path);
    }

    $headers = fgetcsv($fp);
    if ($headers === false) {
        throw new RuntimeException('CSV file is empty: ' . $path);
    }

    $rows = [];
    while (($row = fgetcsv($fp)) !== false) {
        if (count($row) === 1 && $row[0] === null) {
            continue;
        }
        $assoc = [];
        foreach ($headers as $idx => $key) {
            $assoc[$key] = $row[$idx] ?? null;
        }
        $rows[] = $assoc;
    }

    fclose($fp);
    return $rows;
}

function buildItemsFromRow(array $row): array
{
    $items = [];
    $skuMap = [
        'sku_3508_pallets' => ['sku' => '3508', 'target' => 'uv'],
        'sku_3508_unidades' => ['sku' => '3508', 'target' => 'uc'],
        'sku_4409_pallets' => ['sku' => '4409', 'target' => 'uv'],
        'sku_4409_unidades' => ['sku' => '4409', 'target' => 'uc'],
    ];

    foreach ($skuMap as $key => $cfg) {
        $val = isset($row[$key]) ? (int)$row[$key] : 0;
        if ($val <= 0) {
            continue;
        }
        if (!isset($items[$cfg['sku']])) {
            $items[$cfg['sku']] = ['sku' => $cfg['sku'], 'uv' => 0, 'uc' => 0];
        }
        $items[$cfg['sku']][$cfg['target']] += $val;
    }

    return array_values(array_filter($items, static fn(array $item): bool => ($item['uv'] > 0 || $item['uc'] > 0)));
}

function buildDestinatarioPayload(array $row, string $pedidoCodigo): array
{
    $zona = trim((string)($row['zona'] ?? 'GEN'));
    $scenario = trim((string)($row['scenario_id'] ?? 'S'));
    $cod = strtoupper(preg_replace('/[^A-Z0-9]+/', '', $zona));
    if ($cod === '') {
        $cod = 'ZONA';
    }
    $cod = trunc_str('AUTO-' . $cod, 20);
    $nombre = trunc_str('Destino ' . $zona, 80);
    $razon = trunc_str('Dest ' . $zona, 20);
    $direccion = 'Circuito ' . ($zona !== '' ? $zona : 'Sin zona');
    $telefono = '1100' . str_pad(substr($scenario, -1), 3, '0', STR_PAD_LEFT) . '000';

    return [
        'cod' => $cod,
        'nombre' => $nombre,
        'razon' => $razon,
        'direccion' => $direccion,
        'telefono' => $telefono,
        'referencia' => $pedidoCodigo,
    ];
}

function buildDocumentoPayload(array $row, string $pedidoCodigo): array
{
    $tipo = 'FACTURA';
    $numero = $pedidoCodigo . '-' . ($row['scenario_id'] ?? 'S');
    return [
        'tipo' => trunc_str($tipo, 20),
        'numero' => trunc_str($numero, 64),
    ];
}

function buildStagingRow(array $dest, array $doc, string $fecha, string $batchCode, array $item): array
{
    return [
        'cod' => $dest['cod'],
        'razon_social' => $dest['razon'],
        'nombre_destinatario' => $dest['nombre'],
        'direccion' => $dest['direccion'],
        'telefono' => (string)$dest['telefono'],
        'sku_cliente' => $item['sku'],
        'uv_cajas' => $item['uv'],
        'uc_sueltas' => $item['uc'],
        'numero_doc' => $doc['numero'],
        'tipo_factura' => $doc['tipo'],
        'fecha_salida' => $fecha,
        'pre_embarque' => $batchCode,
    ];
}

function trunc_str(?string $s, int $max): string
{
    $s = (string)$s;
    if ($max <= 0) {
        return '';
    }
    if (function_exists('mb_substr')) {
        return mb_substr($s, 0, $max);
    }
    return substr($s, 0, $max);
}
