<?php
// scripts/test_cumplimiento_endpoint.php
declare(strict_types=1);

require_once __DIR__ . '/../config/db.php';

echo "🧪 Testing Cumplimiento Endpoint\n";
echo "================================\n\n";

$host = 'http://localhost:8080';
$url = $host . '/api/reportes/embarque_cumplimiento.php';

// Test 1: Basic request
echo "1. Testing basic request...\n";
$params = [
    'fecha_desde' => date('Y-m-d', strtotime('-7 days')),
    'fecha_hasta' => date('Y-m-d')
];

$queryString = http_build_query($params);
$fullUrl = $url . '?' . $queryString;

echo "   URL: $fullUrl\n";

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $fullUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 30);
curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);

// Set session cookie if needed (mock auth)
$sessionId = 'test_session_' . time();
curl_setopt($ch, CURLOPT_COOKIE, "PHPSESSID=$sessionId");

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

echo "   HTTP Code: $httpCode\n";

if ($response === false) {
    echo "   ❌ Request failed\n";
    exit(1);
}

$data = json_decode($response, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    echo "   ❌ Invalid JSON response\n";
    echo "   Response: " . substr($response, 0, 500) . "\n";
    exit(1);
}

if (isset($data['ok']) && $data['ok'] === true) {
    echo "   ✅ Success\n";
    echo "   📊 Found " . count($data['data']) . " embarques\n";
    
    if (isset($data['metrics'])) {
        echo "   📈 Metrics:\n";
        echo "      - Total embarques: " . $data['metrics']['total_embarques'] . "\n";
        echo "      - Rate cumplimiento: " . $data['metrics']['fulfillment_rates']['rate_cumplimiento'] . "%\n";
        echo "      - Rate entrega parcial: " . $data['metrics']['fulfillment_rates']['rate_entrega_parcial'] . "%\n";
        echo "      - Rate puntualidad: " . $data['metrics']['tiempos']['rate_puntualidad'] . "%\n";
    }
    
    if (isset($data['filter_options'])) {
        echo "   🔧 Filter options:\n";
        echo "      - Estados: " . count($data['filter_options']['estados']) . "\n";
        echo "      - Clientes: " . count($data['filter_options']['clientes']) . "\n";
        echo "      - Destinatarios: " . count($data['filter_options']['destinatarios']) . "\n";
        echo "      - Móviles: " . count($data['filter_options']['moviles']) . "\n";
    }
} else {
    echo "   ❌ API returned error\n";
    echo "   Error: " . ($data['error'] ?? 'Unknown error') . "\n";
    exit(1);
}

echo "\n";

// Test 2: PDF endpoint
echo "2. Testing PDF endpoint...\n";
$pdfUrl = $host . '/api/reportes/embarque_cumplimiento_pdf.php?' . $queryString;
echo "   URL: $pdfUrl\n";

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $pdfUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 30);
curl_setopt($ch, CURLOPT_NOBODY, true); // HEAD request only
curl_setopt($ch, CURLOPT_COOKIE, "PHPSESSID=$sessionId");

curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$contentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
curl_close($ch);

echo "   HTTP Code: $httpCode\n";
echo "   Content Type: $contentType\n";

if ($httpCode === 200 && strpos($contentType, 'pdf') !== false) {
    echo "   ✅ PDF endpoint working\n";
} else {
    echo "   ⚠️ PDF endpoint issue (code: $httpCode, type: $contentType)\n";
}

echo "\n";

// Test 3: Database tables existence
echo "3. Testing database tables...\n";
try {
    $pdo = getPDO();
    
    $tables = [
        'so_embarque',
        'so_embarque_estado', 
        'so_embarque_parada',
        'so_embarque_pre',
        'so_preembarque_item',
        'para_destinatarios',
        'para_clientes',
        'para_moviles',
        'para_choferes',
        'wh_deposito'
    ];
    
    foreach ($tables as $table) {
        $stmt = $pdo->query("SELECT COUNT(*) FROM $table LIMIT 1");
        $count = $stmt->fetchColumn();
        echo "   ✅ Table $table exists (rows: $count)\n";
    }
    
} catch (Exception $e) {
    echo "   ❌ Database error: " . $e->getMessage() . "\n";
    exit(1);
}

echo "\n🎉 All tests completed successfully!\n";
echo "📋 The Cumplimiento de Entregas report is ready to use.\n\n";

echo "🔗 URLs to test in browser:\n";
echo "   - Main report: $host/reportes/embarques_cumplimiento\n";
echo "   - API endpoint: $url\n";
echo "   - PDF export: " . str_replace('.php', '_pdf.php', $url) . "\n";