<?php
require_once 'config/db.php';

echo "=== PRUEBA DIRECTA DEL STORED PROCEDURE ===\n\n";

try {
    $pdo = get_pdo();
    
    // 1. Obtener el código del pedido SO-28
    $stmt = $pdo->prepare('SELECT codigo FROM so_pedido WHERE id = ?');
    $stmt->execute([28]);
    $soCodigo = $stmt->fetchColumn();
    
    if (!$soCodigo) {
        echo "❌ No se encontró el pedido SO-28\n";
        exit;
    }
    
    echo "Código del pedido SO-28: $soCodigo\n\n";
    
    // 2. Configurar parámetros del stored procedure
    $parametros = [
        'p_pedido_codigo' => $soCodigo,
        'p_deposito_code' => 'DEP1', 
        'p_pos_prep_code' => null,  // Let it auto-select
        'p_simulate' => 1  // Simulación primero
    ];
    
    echo "Parámetros para el SP:\n";
    foreach ($parametros as $key => $value) {
        echo "  $key: " . ($value === null ? 'NULL' : $value) . "\n";
    }
    echo "\n";
    
    // 3. Configurar timeouts
    $pdo->exec('SET SESSION innodb_lock_wait_timeout = 120');
    $pdo->exec('SET SESSION lock_wait_timeout = 120');
    $pdo->exec('SET SESSION transaction_isolation = "READ-COMMITTED"');
    $pdo->exec('SET SESSION autocommit = 1');
    echo "✅ Timeouts y configuración establecidos\n\n";
    
    // 4. Ejecutar el stored procedure
    echo "Ejecutando stored procedure...\n";
    $start = microtime(true);
    
    $stmt = $pdo->prepare('CALL sp_so_preparar_auto(?, ?, ?, ?)');
    $stmt->execute([
        $parametros['p_pedido_codigo'],
        $parametros['p_deposito_code'], 
        $parametros['p_pos_prep_code'],
        $parametros['p_simulate']
    ]);
    
    $duration = microtime(true) - $start;
    echo "✅ Stored procedure ejecutado en " . round($duration, 2) . " segundos\n\n";
    
    // 5. Procesar resultados
    $sets = [];
    $resultSetCount = 0;
    
    do {
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if ($rows !== false && !empty($rows)) {
            $sets[] = $rows;
            $resultSetCount++;
            echo "ResultSet $resultSetCount: " . count($rows) . " filas\n";
        }
    } while ($stmt->nextRowset());
    
    echo "\nTotal de result sets: $resultSetCount\n\n";
    
    // 6. Mostrar plan de simulación si existe
    if (!empty($sets) && $parametros['p_simulate'] == 1) {
        echo "=== PLAN DE SIMULACIÓN ===\n";
        $plan = $sets[0] ?? [];
        if (!empty($plan)) {
            foreach ($plan as $i => $step) {
                echo "Paso " . ($i + 1) . ":\n";
                echo "  Fase: {$step['fase']}\n";
                echo "  Item ID: {$step['item_id']}\n";
                echo "  Producto: {$step['producto_id']}\n";
                echo "  From: {$step['from_pos_id']} → To: {$step['to_pos_id']}\n";
                echo "  Cantidad: {$step['take_uv']} UV + {$step['take_uc']} UC\n";
                echo "  Lote: {$step['lote_code']} (Venc: {$step['venc']})\n\n";
            }
        } else {
            echo "  ⚠️ No se generó plan (posiblemente no hay items pendientes)\n";
        }
    }
    
    // 7. Mostrar resumen final
    if (!empty($sets)) {
        $resumen = end($sets);
        echo "=== RESUMEN FINAL ===\n";
        foreach ($resumen as $item) {
            echo "Item {$item['pedido_dest_item_id']}:\n";
            echo "  Producto: {$item['producto_id']}\n";
            echo "  Expected: {$item['expected_uv']} UV + {$item['expected_uc']} UC\n";
            echo "  Prepared: {$item['prepared_uv']} UV + {$item['prepared_uc']} UC\n";
            echo "  Pendiente: {$item['pend_uv']} UV + {$item['pend_uc']} UC\n\n";
        }
    }
    
} catch (Throwable $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "Código: " . $e->getCode() . "\n";
    echo "Archivo: " . $e->getFile() . ":" . $e->getLine() . "\n";
}
?>