<?php
/**
 * Script simple para probar UV→UC con un pedido específico
 */
declare(strict_types=1);

$ROOT = dirname(__DIR__);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

// Configurar el ID del pedido a probar
$soId = 16; // Cambiar por el ID correcto
$simulateMode = true; // true = no hace cambios reales

try {
    $pdo = get_pdo();
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    echo "🔍 DIAGNÓSTICO UV→UC PARA PEDIDO ID: $soId\n";
    echo str_repeat("=", 60) . "\n\n";
    
    // 1. Verificar que el pedido existe y obtener su código
    $stmt = $pdo->prepare('SELECT codigo, estado_id FROM so_pedido WHERE id = ?');
    $stmt->execute([$soId]);
    $pedido = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$pedido) {
        echo "❌ Error: No se encontró el pedido con ID $soId\n";
        echo "💡 Sugerencia: Cambia \$soId en la línea 9 del script\n\n";
        
        // Mostrar pedidos disponibles
        echo "📋 Pedidos disponibles:\n";
        $pedidos = $pdo->query("SELECT id, codigo, estado_id FROM so_pedido ORDER BY id DESC LIMIT 10")->fetchAll(PDO::FETCH_ASSOC);
        foreach ($pedidos as $p) {
            echo "   ID: {$p['id']} | Código: {$p['codigo']} | Estado: {$p['estado_id']}\n";
        }
        exit;
    }
    
    $soCodigo = $pedido['codigo'];
    echo "✅ Pedido encontrado: $soCodigo (Estado: {$pedido['estado_id']})\n\n";
    
    // 2. Verificar items del pedido y factores de conversión
    echo "📊 ANÁLISIS DE ITEMS Y CONVERSIONES:\n";
    echo str_repeat("-", 50) . "\n";
    
    $items = $pdo->prepare("
        SELECT 
            i.id as item_id,
            i.producto_id,
            p.sku,
            p.denominacion,
            i.expected_uv,
            i.expected_uc,
            i.prepared_uv,
            i.prepared_uc,
            (i.expected_uv - i.prepared_uv) as pending_uv,
            (i.expected_uc - i.prepared_uc) as pending_uc,
            pp.unidades_por_uv
        FROM so_pedido_dest d
        JOIN so_pedido_dest_item i ON i.pedido_dest_id = d.id
        JOIN para_productos p ON p.id = i.producto_id
        LEFT JOIN para_producto_pack pp ON pp.producto_id = i.producto_id
        WHERE d.pedido_id = ?
        ORDER BY i.id
    ");
    $items->execute([$soId]);
    $itemsData = $items->fetchAll(PDO::FETCH_ASSOC);
    
    $hasUcDeficit = false;
    $hasConversionFactor = false;
    
    foreach ($itemsData as $item) {
        $pendingUc = max(0, (int)$item['pending_uc']);
    $ucPerUv = (int)($item['unidades_por_uv'] ?? 0);
        
        echo sprintf(
            "Item %s: %s\n",
            $item['item_id'],
            $item['sku']
        );
        echo sprintf(
            "  UV: %s/%s (pendiente: %s) | UC: %s/%s (pendiente: %s)\n",
            $item['prepared_uv'] ?? 0,
            $item['expected_uv'] ?? 0,
            max(0, (int)$item['expected_uv'] - (int)$item['prepared_uv']),
            $item['prepared_uc'] ?? 0,
            $item['expected_uc'] ?? 0,
            $pendingUc
        );
        echo sprintf(
            "  Factor conversión: %s UC por UV %s\n",
            $ucPerUv,
            $ucPerUv > 0 ? "✅" : "❌ (REQUERIDO PARA UV→UC)"
        );
        echo "\n";
        
        if ($pendingUc > 0) $hasUcDeficit = true;
        if ($ucPerUv > 0) $hasConversionFactor = true;
    }
    
    // 3. Verificar stock disponible
    echo "📦 STOCK DISPONIBLE:\n";
    echo str_repeat("-", 50) . "\n";
    
    foreach ($itemsData as $item) {
        $prodId = $item['producto_id'];
        
        // Stock en PICKING
        $stockPick = $pdo->prepare("
            SELECT COALESCE(SUM(qty_uv),0) as uv, COALESCE(SUM(qty_uc),0) as uc
            FROM wh_stock s
            JOIN wh_posicion pos ON pos.id = s.posicion_id
            JOIN wh_ambiente amb ON amb.id = pos.ambiente_id AND amb.code = 'PICKING'
            WHERE s.producto_id = ? AND s.deposito_id = 1
        ");
        $stockPick->execute([$prodId]);
        $pick = $stockPick->fetch(PDO::FETCH_ASSOC);
        
        echo sprintf(
            "Producto %s: PICKING UV:%s UC:%s\n",
            $item['sku'],
            $pick['uv'] ?? 0,
            $pick['uc'] ?? 0
        );
    }
    echo "\n";
    
    // 4. Condiciones para UV→UC
    echo "🔍 EVALUACIÓN DE CONDICIONES UV→UC:\n";
    echo str_repeat("-", 50) . "\n";
    echo "✅ Checkbox UV→UC activado: Simulado como SÍ\n";
    echo ($hasConversionFactor ? "✅" : "❌") . " Factor de conversión configurado: " . ($hasConversionFactor ? "SÍ" : "NO") . "\n";
    echo ($hasUcDeficit ? "✅" : "❌") . " Déficit de UC: " . ($hasUcDeficit ? "SÍ" : "NO") . "\n";
    echo "\n";
    
    if (!$hasConversionFactor) {
        echo "🔧 SOLUCIÓN REQUERIDA:\n";
        echo "Para habilitar UV→UC necesitas configurar factores de conversión:\n\n";
        foreach ($itemsData as $item) {
            if (!$item['unidades_por_uv']) {
                echo "-- Ejemplo para producto {$item['sku']} (100 UC por UV):\n";
                echo "INSERT INTO para_producto_pack (producto_id, unidades_por_uv) VALUES ({$item['producto_id']}, 100)\n";
                echo "ON DUPLICATE KEY UPDATE unidades_por_uv = 100;\n\n";
            }
        }
        echo "\n";
    }
    
    // 5. Probar el SP si todo está configurado
    if ($hasConversionFactor && $hasUcDeficit) {
        echo "🧪 EJECUTANDO PRUEBA DEL SP:\n";
        echo str_repeat("-", 50) . "\n";
        
        // Configurar variables
        $pdo->exec('SET @so_pre_log := 1');
        $pdo->exec('SET @so_pre_uc_to_uv := 0');
        $pdo->exec('SET @so_pre_uv_to_uc := 1');
        echo "✅ Variables configuradas (log=1, uc_to_uv=0, uv_to_uc=1)\n";
        
        // Limpiar logs previos
        $pdo->exec("DELETE FROM so_preparar_auto_log WHERE pedido_codigo = '$soCodigo'");
        
        try {
            // Ejecutar SP
            $stmt = $pdo->prepare('CALL sp_so_preparar_auto(?, ?, ?, ?)');
            $stmt->execute([$soCodigo, 'DEP1', null, $simulateMode ? 1 : 0]);
            $stmt->closeCursor();
            
            echo "✅ SP ejecutado exitosamente\n\n";
            
            // Mostrar logs generados
            echo "📋 LOGS GENERADOS:\n";
            echo str_repeat("-", 50) . "\n";
            $logs = $pdo->query("
                SELECT id, fase, msg, take_uv, take_uc
                FROM so_preparar_auto_log 
                WHERE pedido_codigo = '$soCodigo' 
                ORDER BY id ASC
            ")->fetchAll(PDO::FETCH_ASSOC);
            
            if (empty($logs)) {
                echo "❌ No se generaron logs (posible problema)\n";
            } else {
                foreach ($logs as $log) {
                    echo sprintf(
                        "[%s] %s | UV:%s UC:%s | %s\n",
                        $log['id'],
                        $log['fase'] ?? 'N/A',
                        $log['take_uv'] ?? '0',
                        $log['take_uc'] ?? '0',
                        $log['msg'] ?? ''
                    );
                }
                
                // Buscar logs específicos de UV→UC
                $uvucLogs = array_filter($logs, function($log) {
                    return strpos($log['msg'] ?? '', 'UV→UC') !== false;
                });
                
                if (empty($uvucLogs)) {
                    echo "\n⚠️  No se encontraron logs de conversión UV→UC\n";
                    echo "💡 Revisa los logs 'DEBUG' para ver por qué no se activó\n";
                } else {
                    echo "\n✅ Se encontraron " . count($uvucLogs) . " logs de conversión UV→UC\n";
                }
            }
            
        } catch (Exception $e) {
            echo "❌ Error ejecutando SP: " . $e->getMessage() . "\n";
        }
    } else {
        echo "⏭️  Saltando prueba del SP (condiciones no cumplidas)\n";
    }
    
    echo "\n" . str_repeat("=", 60) . "\n";
    echo "✅ Diagnóstico completado\n";
    
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
    echo "Archivo: " . $e->getFile() . ":" . $e->getLine() . "\n";
}