<?php
/**
 * Script para ver logs de debug del SP sp_so_preparar_auto
 * Muestra los logs más recientes para diagnosticar problemas
 */
declare(strict_types=1);

$ROOT = dirname(__DIR__);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

function show_logs() {
    try {
        $pdo = get_pdo();
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        echo "=== Debug Logs para sp_so_preparar_auto ===\n\n";
        
        // Verificar si la tabla de logs existe
        $tables = $pdo->query("SHOW TABLES LIKE 'so_preparar_auto_log'")->fetchAll();
        if (empty($tables)) {
            echo "❌ La tabla 'so_preparar_auto_log' no existe.\n";
            echo "   Necesitas ejecutar las migraciones para crear la tabla de logs.\n";
            return;
        }
        
        // Mostrar logs más recientes
        echo "📋 Logs más recientes (últimos 20):\n";
        echo str_repeat("=", 80) . "\n";
        
        $stmt = $pdo->query("
            SELECT 
                id,
                pedido_codigo,
                fase,
                item_id,
                producto_id,
                lote_code,
                take_uv,
                take_uc,
                msg
            FROM so_preparar_auto_log 
            ORDER BY id DESC 
            LIMIT 20
        ");
        
        $logs = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($logs)) {
            echo "ℹ️  No hay logs disponibles.\n";
            echo "   Esto puede significar:\n";
            echo "   1. No se ha ejecutado el SP con debug=1\n";
            echo "   2. El parámetro @so_pre_log no se está estableciendo\n";
            echo "   3. Los logs se están limpiando automáticamente\n\n";
        } else {
            foreach ($logs as $log) {
                $fase = $log['fase'] ?? 'N/A';
                $pedido = $log['pedido_codigo'] ?? 'N/A';
                $lote = $log['lote_code'] ?? 'N/A';
                $uv = $log['take_uv'] ?? '0';
                $uc = $log['take_uc'] ?? '0';
                $msg = $log['msg'] ?? '';
                
                echo sprintf(
                    "[%s] %s | %s | UV:%s UC:%s | %s\n",
                    $log['id'],
                    $fase,
                    $pedido,
                    $uv,
                    $uc,
                    $msg
                );
            }
        }
        
        echo "\n" . str_repeat("=", 80) . "\n";
        
        // Buscar logs específicos de UV→UC
        echo "🔍 Logs específicos de UV→UC:\n";
        $uvToUcLogs = $pdo->query("
            SELECT * FROM so_preparar_auto_log 
            WHERE msg LIKE '%UV→UC%' OR msg LIKE '%uv_to_uc%'
            ORDER BY id DESC 
            LIMIT 10
        ")->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($uvToUcLogs)) {
            echo "❌ No se encontraron logs de UV→UC.\n";
            echo "   Esto indica que:\n";
            echo "   1. La conversión UV→UC no se está ejecutando\n";
            echo "   2. Los parámetros no llegan correctamente al SP\n";
            echo "   3. Las condiciones para UV→UC no se cumplen\n\n";
        } else {
            foreach ($uvToUcLogs as $log) {
                echo "✅ " . ($log['msg'] ?? '') . "\n";
            }
        }
        
        // Verificar variables de sesión
        echo "\n🔧 Variables de sesión actuales:\n";
        $sessionVars = $pdo->query("
            SELECT 
                @@session.autocommit as autocommit,
                @@session.transaction_isolation as isolation
        ")->fetch(PDO::FETCH_ASSOC);
        
        foreach ($sessionVars as $var => $value) {
            echo "   $var: $value\n";
        }
        
        // Verificar si hay preembarques recientes
        echo "\n📦 Preembarques recientes:\n";
        $preembarques = $pdo->query("
            SELECT codigo, estado_id 
            FROM so_preembarque 
            WHERE codigo LIKE 'PRE-%' 
            ORDER BY id DESC 
            LIMIT 5
        ")->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($preembarques)) {
            echo "❌ No hay preembarques recientes.\n";
        } else {
            foreach ($preembarques as $pre) {
                echo "   " . $pre['codigo'] . " (Estado: " . $pre['estado_id'] . ")\n";
            }
        }
        
    } catch (PDOException $e) {
        if (strpos($e->getMessage(), 'Connection refused') !== false || 
            strpos($e->getMessage(), 'No se puede establecer una conexión') !== false) {
            echo "❌ No se puede conectar a la base de datos.\n";
            echo "   Error: Conexión rechazada\n";
            echo "   Solución: Inicia el servidor MySQL/MariaDB\n\n";
            
            // Mostrar comando para ver logs en entorno de desarrollo
            echo "🔧 Para ver logs en entorno de desarrollo:\n";
            echo "   1. Activa el servidor de base de datos\n";
            echo "   2. Ejecuta: php scripts/view_debug_logs.php\n";
            echo "   3. O usa la interfaz web con debug=1\n\n";
            
            echo "🔍 Para diagnosticar sin BD:\n";
            echo "   1. Revisa que el frontend envía uv_to_uc=1\n";
            echo "   2. Verifica que el endpoint procesa el parámetro\n";
            echo "   3. Confirma que el SP recibe @so_pre_uv_to_uc\n\n";
            
        } else {
            echo "❌ Error de base de datos: " . $e->getMessage() . "\n";
        }
    } catch (Throwable $e) {
        echo "❌ Error: " . $e->getMessage() . "\n";
        echo "   Archivo: " . $e->getFile() . ":" . $e->getLine() . "\n";
    }
}

// Función para limpiar logs antiguos
function clean_old_logs() {
    try {
        $pdo = get_pdo();
        $result = $pdo->exec("DELETE FROM so_preparar_auto_log WHERE created_at < DATE_SUB(NOW(), INTERVAL 1 DAY)");
        echo "🧹 Logs antiguos limpiados: $result registros eliminados\n";
    } catch (Throwable $e) {
        echo "❌ Error limpiando logs: " . $e->getMessage() . "\n";
    }
}

// Función para forzar un test con logs
function force_test() {
    try {
        $pdo = get_pdo();
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        echo "🧪 Ejecutando test forzado con logs...\n";
        
        // Configurar variables
        $pdo->exec('SET @so_pre_log := 1');
        $pdo->exec('SET @so_pre_uc_to_uv := 0');
        $pdo->exec('SET @so_pre_uv_to_uc := 1');
        
        echo "✅ Variables configuradas\n";
        echo "   @so_pre_log = 1\n";
        echo "   @so_pre_uc_to_uv = 0\n";
        echo "   @so_pre_uv_to_uc = 1\n\n";
        
        // Buscar un pedido válido para probar
        $pedidoTest = null;
        $pedidos = $pdo->query("
            SELECT codigo FROM so_pedido 
            WHERE estado_id IS NOT NULL 
            ORDER BY id DESC 
            LIMIT 5
        ")->fetchAll(PDO::FETCH_ASSOC);
        
        if (!empty($pedidos)) {
            $pedidoTest = $pedidos[0]['codigo'];
            echo "📋 Usando pedido de prueba: $pedidoTest\n";
        } else {
            echo "❌ No se encontraron pedidos para probar\n";
            return;
        }
        
        // Limpiar logs previos
        $pdo->exec("DELETE FROM so_preparar_auto_log WHERE pedido_codigo = '$pedidoTest'");
        
        // Ejecutar SP
        $stmt = $pdo->prepare('CALL sp_so_preparar_auto(?, ?, ?, ?)');
        $stmt->execute([$pedidoTest, 'DEP1', null, 1]); // simulate=1
        
        // Cerrar statement
        $stmt->closeCursor();
        
        echo "✅ SP ejecutado con éxito\n\n";
        
        // Mostrar logs generados
        show_logs();
        
    } catch (Throwable $e) {
        echo "❌ Error en test forzado: " . $e->getMessage() . "\n";
    }
}

// Detectar parámetros de línea de comandos
$action = $argv[1] ?? 'show';

switch ($action) {
    case 'clean':
        clean_old_logs();
        break;
    case 'test':
        force_test();
        break;
    case 'show':
    default:
        show_logs();
        break;
}

echo "\n💡 Uso:\n";
echo "   php scripts/view_debug_logs.php show   # Mostrar logs (default)\n";
echo "   php scripts/view_debug_logs.php test   # Ejecutar test forzado\n";
echo "   php scripts/view_debug_logs.php clean  # Limpiar logs antiguos\n";