<?php
/**
 * DataTables Editor — DB config bridge
 *
 * Este archivo define $sql_details para que el Editor PHP conecte a la base
 * de datos usando las variables de entorno de tu proyecto (.env).
 *
 * Variables esperadas en .env:
 *   APP_ENV=development|production
 *   DB_CONNECTION=mysql        ; mysql|pgsql|sqlsrv|sqlite|oracle
 *   DB_HOST=localhost
 *   DB_PORT=3306
 *   DB_DATABASE=sol
 *   DB_USERNAME=root
 *   DB_PASSWORD=
 *   DB_CHARSET=utf8mb4
 *
 * Notas:
 * - No hardcodeamos credenciales.
 * - Intentamos cargar Dotenv si está disponible; si no, usamos $_ENV/getenv().
 * - Ajustamos 'type' a los nombres esperados por Editor: Mysql|Postgres|Sqlserver|Sqlite|Oracle
 */

// Asegura que DataTables.php haya sido incluido
if (!defined('DATATABLES')) {
    exit(1);
}

// -------------------------------------------------------------
// Cargar .env (en caso de que aún no esté cargado)
// -------------------------------------------------------------
$__root = isset($_SERVER['DOCUMENT_ROOT']) && is_dir($_SERVER['DOCUMENT_ROOT'])
    ? rtrim($_SERVER['DOCUMENT_ROOT'], DIRECTORY_SEPARATOR)
    : dirname(__DIR__, 3); // fallback: .../vendor/Editor/lib -> sube 3 niveles al proyecto

// Try Composer autoload (por si Dotenv está instalado vía composer)
$__autoload1 = $__root . '/vendor/autoload.php';
if (is_file($__autoload1)) {
    /** @noinspection PhpIncludeInspection */
    @require_once $__autoload1;
}

// Carga segura de dotenv si existe
if (class_exists('Dotenv\Dotenv')) {
    try {
        $dotenv = Dotenv\Dotenv::createImmutable($__root);
        // safeLoad: no pisa lo ya existente y no lanza excepción si falta el archivo
        $dotenv->safeLoad();
    } catch (\Throwable $e) {
        // Silenciar: si falla, seguiremos leyendo de $_ENV/getenv
    }
}

// Helper env()
if (!function_exists('editor_env')) {
    function editor_env(string $key, $default = null) {
        // Prioriza $_ENV (cargado por Dotenv) y luego getenv()
        if (array_key_exists($key, $_ENV)) {
            return $_ENV[$key];
        }
        $v = getenv($key);
        return ($v === false || $v === null) ? $default : $v;
    }
}

// -------------------------------------------------------------
// Error reporting según APP_ENV
// -------------------------------------------------------------
$appEnv = strtolower((string) editor_env('APP_ENV', 'development'));
if ($appEnv === 'production') {
    error_reporting(0);
    ini_set('display_errors', '0');
} else {
    error_reporting(E_ALL);
    ini_set('display_errors', '1');
}

// -------------------------------------------------------------
// Mapear tipo de conexión
// -------------------------------------------------------------
$driver = strtolower((string) editor_env('DB_CONNECTION', 'mysql'));
switch ($driver) {
    case 'mysql':   $type = 'Mysql';    break;
    case 'pgsql':
    case 'postgres':$type = 'Postgres'; break;
    case 'sqlsrv':
    case 'mssql':   $type = 'Sqlserver';break;
    case 'sqlite':  $type = 'Sqlite';   break;
    case 'oracle':  $type = 'Oracle';   break;
    default:        $type = 'Mysql';    break;
}

// -------------------------------------------------------------
// Leer credenciales desde .env
// -------------------------------------------------------------
$dbHost   = (string) editor_env('DB_HOST', 'localhost');
$dbPort   = (string) editor_env('DB_PORT', '');
$dbName   = (string) editor_env('DB_DATABASE', '');
$dbUser   = (string) editor_env('DB_USERNAME', 'root');
$dbPass   = (string) editor_env('DB_PASSWORD', '');
$dbCharset= (string) editor_env('DB_CHARSET', 'utf8mb4');

// DSN extra — charset para MySQL es recomendable
$dsnExtra = '';
if ($type === 'Mysql' && $dbCharset) {
    $dsnExtra = 'charset=' . $dbCharset;
}

// Atributos PDO por defecto (puedes extenderlos desde tu config global si hace falta)
$pdoAttr = [
    \PDO::ATTR_ERRMODE            => \PDO::ERRMODE_EXCEPTION,
    \PDO::ATTR_DEFAULT_FETCH_MODE => \PDO::FETCH_ASSOC,
];

// -------------------------------------------------------------
// Definir $sql_details para Editor
// -------------------------------------------------------------
$sql_details = [
    'type'   => $type,     // Mysql|Postgres|Sqlserver|Sqlite|Oracle
    'user'   => $dbUser,
    'pass'   => $dbPass,
    'host'   => $dbHost,
    'port'   => $dbPort,
    'db'     => $dbName,
    'dsn'    => $dsnExtra, // e.g. 'charset=utf8mb4'
    'pdoAttr'=> $pdoAttr,
];

// -------------------------------------------------------------
// (Opcional) Entorno de desarrollo de DataTables — NO aplicable aquí
// -------------------------------------------------------------
// Eliminado el include de /datatables/pdo.php del ejemplo original
