<?php

declare(strict_types=1);

/**
 * SOL - Sistema de Operaciones Logísticas
 * Vista: views/control/conteo/index.php
 * Reporte: Conteo por ubicación (layout de depósito)
 *
 * Notas:
 * - La cabecera (filtros y controles) se mueve a esta vista para que
 *   el parcial warehouse/rack_layout.php quede reutilizable (sin toolbar).
 * - Se pasan $options a rack_layout con show_toolbar=false.
 * - Se leen parámetros de la URL para inicializar (opcional):
 *     ?fecha=YYYY-MM-DD&deposito_id=1&rack=5&view=microtiles
 */

$BASE = dirname(__DIR__, 3);

require_once $BASE . '/app/Support/ViewHelpers.php';
require_once $BASE . '/app/Support/Assets.php';

// -------------------------
// Meta
// -------------------------
sol_section('page_title', 'Control · Conteo por ubicación');

// -------------------------
// Lee params (opcionales) para inicializar la vista
// -------------------------
$fechaParam      = isset($_GET['fecha']) ? (string)$_GET['fecha'] : '';
$depositoIdParam = isset($_GET['deposito_id']) ? (int)$_GET['deposito_id'] : 0;
$rackParam       = isset($_GET['rack']) ? (string)$_GET['rack'] : '';
$viewParam       = isset($_GET['view']) ? (string)$_GET['view'] : 'microtiles';

// Normaliza rack aceptando "5" o "R05"
$rackNum = 0;
if ($rackParam !== '') {
  if (preg_match('/^\s*[Rr]\s*0*([0-9]+)\s*$/', $rackParam, $m)) {
    $rackNum = (int)$m[1];
  } else {
    $rackNum = (int)$rackParam;
  }
}

// Estructuras mínimas para el parcial (pueden quedar vacías)
$deposito = ($depositoIdParam > 0)
  ? ['id' => $depositoIdParam, 'code' => 'DEP' . $depositoIdParam, 'name' => null]
  : ['id' => null, 'code' => '', 'name' => null];

$rack = ($rackNum > 0)
  ? ['id' => $rackNum, 'code' => 'R' . str_pad((string)$rackNum, 2, '0', STR_PAD_LEFT), 'name' => null]
  : ['id' => null, 'code' => '', 'name' => null];

// Sin posiciones server-side: las cargará el JS por API cuando el usuario elija depósito/rack
$positions = [];

// -------------------------
// Assets específicos
// -------------------------
// 1) CSS: layout de depósito (warehouse.scss compilado a warehouse.css)
// 2) JS: primero core, luego renderers, por último bootstrap de la página
sol_assets_push(
  [
    url('assets/css/warehouse.css'),
  ],
  [
    url('page-scripts/warehouse/layout-core.js'),
    url('page-scripts/warehouse/render-microtiles.js'),
    url('page-scripts/warehouse/render-shelf-strips.js'),
    url('page-scripts/warehouse/render-column-cards.js'),
    url('page-scripts/pages/control/conteo.js'),
  ]
);

// -------------------------
// Contenido principal
// -------------------------
ob_start();
?>
<div class="container-fluid">

  <!-- Cabecera de filtros / controles (ahora vive en la vista, no en el parcial) -->
  <div class="card mb-3">
    <div class="card-body">
      <form id="frmConteo" class="row g-3">
        <!-- Fecha de conteo -->
        <div class="col-12 col-md-3">
          <label for="flt-fecha" class="form-label">Fecha</label>
          <input
            type="date"
            id="flt-fecha"
            name="fecha"
            class="form-control"
            value="<?= htmlspecialchars($fechaParam !== '' ? $fechaParam : date('Y-m-d')) ?>" />
        </div>

        <!-- Depósito (se cargará por API meta=depositos) -->
        <div class="col-12 col-md-3">
          <label for="flt-deposito" class="form-label">Depósito</label>
          <select id="flt-deposito" name="deposito_id" class="form-select" data-source="api/control/conteo_layout.php?meta=depositos">
            <?php if ($deposito['id']): ?>
              <option value="<?= (int)$deposito['id'] ?>" selected>DEP<?= (int)$deposito['id'] ?></option>
            <?php else: ?>
              <option value="">Seleccione...</option>
            <?php endif; ?>
          </select>
        </div>

        <!-- Rack (depende de depósito: api/control/conteo_layout.php?meta=racks&deposito_id=###) -->
        <div class="col-12 col-md-3">
          <label for="flt-rack" class="form-label">Rack</label>
          <select id="flt-rack" name="rack" class="form-select" data-source-base="api/control/conteo_layout.php?meta=racks&deposito_id=">
            <?php if ($rack['id']): ?>
              <option value="<?= htmlspecialchars($rack['code']) ?>" selected><?= htmlspecialchars($rack['code']) ?></option>
            <?php else: ?>
              <option value="">Seleccione...</option>
            <?php endif; ?>
          </select>
        </div>

        <!-- Modo de vista -->
        <div class="col-12 col-md-2">
          <label for="flt-view" class="form-label">Vista</label>
          <select id="flt-view" name="view" class="form-select" data-role="layout-mode">
            <?php
            $view = strtolower($viewParam) ?: 'microtiles';
            $opts = [
              'microtiles'    => 'Microtiles',
              'shelf-strips'  => 'Shelf strips',   // ← antes 'shelf_strips'
              'column-cards'  => 'Column cards',   // ← antes 'column_cards'
              'cubes'         => 'Cubes',
            ];
            foreach ($opts as $k => $label) {
              $sel = ($k === $view) ? 'selected' : '';
              echo "<option value=\"{$k}\" {$sel}>{$label}</option>";
            }
            ?>
          </select>

        </div>

        <!-- Mostrar leyenda -->
        <div class="col-12 col-md-1">
          <div class="form-check mt-4">
            <input class="form-check-input" type="checkbox" value="1" id="chk-legend" checked>
            <label class="form-check-label" for="chk-legend">Leyenda</label>
          </div>
        </div>

        <!-- Botón aplicar -->
        <div class="col-12 col-md-12 d-flex gap-2 justify-content-end">
          <button type="submit" class="btn btn-primary">Aplicar</button>
        </div>
      </form>
    </div>
  </div>

  <?php
  // Render del canvas reutilizable del rack, sin toolbar interna
  $options = [
    'mode'         => $view,
    'show_toolbar' => false, // IMPORTANTE: toolbar ahora está en la vista
    'title'      => 'Conteo' // (opcional) si en algún momento querés mostrar un título interno
  ];

  include project_path('views/layout/_partials/warehouse/rack_layout.php');
  ?>
  <!-- Modal: Conteo de posición -->
  <div class="modal fade" id="mdlConteoPos" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-xl modal-dialog-scrollable">
      <div class="modal-content">
        <div class="modal-header">
          <div>
            <h5 class="modal-title">Conteo en posición</h5>
            <div class="small text-muted" id="mdl-conteo-pos-subtitle"><!-- DEP-RK-Cxx-Nxx-Fyy --></div>
          </div>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
        </div>

        <div class="modal-body">
          <!-- Cabecera pallet / posición -->
          <div id="mdl-conteo-pos-header" class="mb-3">
            <!-- Se completa por JS -->
          </div>

          <!-- Items esperados en la posición -->
          <div class="table-responsive">
            <table class="table table-sm align-middle" id="tblConteoItems">
              <thead>
                <tr>
                  <th>SKU</th>
                  <th>Producto</th>
                  <th>Lote</th>
                  <th>Venc.</th>
                  <th class="text-end">UC esperadas</th>
                  <th class="text-end">UV esperadas</th>
                  <th class="text-end">UC contadas</th>
                  <th class="text-end">UV contadas</th>
                  <th>Obs.</th>
                </tr>
              </thead>
              <tbody><!-- JS --></tbody>
              <tfoot>
                <tr class="fw-semibold">
                  <td colspan="4" class="text-end">Totales</td>
                  <td class="text-end" id="t_uc_exp">0</td>
                  <td class="text-end" id="t_uv_exp">0</td>
                  <td class="text-end" id="t_uc_cnt">0</td>
                  <td class="text-end" id="t_uv_cnt">0</td>
                  <td></td>
                </tr>
              </tfoot>
            </table>
          </div>
        </div>

        <div class="modal-footer">
          <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cerrar</button>
          <button type="button" class="btn btn-primary" id="btnGuardarConteo" disabled>Guardar conteo</button>
        </div>
      </div>
    </div>
  </div>

</div>
<?php
$content = ob_get_clean();

// -------------------------
// Render del layout
// -------------------------
include project_path('views/layout/layout_app.php');
