<?php
declare(strict_types=1);
/**
 * Sección: Gráficos/resúmenes
 * Placeholder para tu librería de gráficos (o KPIs adicionales)
 */

if (!function_exists('project_path')) {
  function project_path(string $rel): string {
    return rtrim(dirname(__DIR__, 4), DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . ltrim($rel, DIRECTORY_SEPARATOR);
  }
}

require_once project_path('config/db.php');

$pdo = null;
try {
  $pdo = getPDO();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (Throwable $e) {
  $pdo = null;
}

$ocupacionDepositos = [];
$ocupacionTotales = [
  'capacidad_pallets'   => 0,
  'pallets_actuales'    => 0,
  'pallets_disponibles' => 0,
  'porcentaje'          => 0.0,
  'm2_totales'          => 0.0,
  'm2_ocupados'         => 0.0,
  'm2_disponibles'      => 0.0,
];
$ocupacionError = null;
$ocupacionNota  = null;

if ($pdo instanceof PDO) {
  try {
    $hasTable = function(PDO $pdoConn, string $tbl): bool {
      static $cache = [];
      if (array_key_exists($tbl, $cache)) {
        return $cache[$tbl];
      }
      $st = $pdoConn->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?');
      $st->execute([$tbl]);
      return $cache[$tbl] = ((int)$st->fetchColumn() > 0);
    };

    $hasColumn = function(PDO $pdoConn, string $tbl, string $col) use ($hasTable): bool {
      static $cache = [];
      $key = $tbl . '.' . $col;
      if (array_key_exists($key, $cache)) {
        return $cache[$key];
      }
      if (!$hasTable($pdoConn, $tbl)) {
        return $cache[$key] = false;
      }
      $st = $pdoConn->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?');
      $st->execute([$tbl, $col]);
      return $cache[$key] = ((int)$st->fetchColumn() > 0);
    };

    $depositTable = $hasTable($pdo, 'wh_deposito') ? 'wh_deposito' : null;
    $positionTable = null;
    foreach (['wh_posicion', 'wh_posiciones'] as $candidate) {
      if ($hasTable($pdo, $candidate)) {
        $positionTable = $candidate;
        break;
      }
    }

    if ($depositTable && $positionTable) {
      $defaultM2 = 1.2;
      $hasM2Column = $hasColumn($pdo, $positionTable, 'm2_por_pallet');
      if (!$hasM2Column) {
        $ocupacionNota = 'm2 estimados con un factor de 1,20 por pallet (columna wh_posicion.m2_por_pallet ausente).';
      }

      $posJoinConditions = ['p.deposito_id = d.id'];
      if ($hasColumn($pdo, $positionTable, 'deleted_at')) {
        $posJoinConditions[] = '(p.deleted_at IS NULL)';
      }
      if ($hasColumn($pdo, $positionTable, 'activo')) {
        $posJoinConditions[] = '(p.activo IS NULL OR p.activo = 1)';
      }
      $posJoin = 'LEFT JOIN ' . $positionTable . ' p ON ' . implode(' AND ', $posJoinConditions);

      $depWhere = ['1=1'];
      $depWherePlain = ['1=1'];
      if ($hasColumn($pdo, $depositTable, 'deleted_at')) {
        $depWhere[] = '(d.deleted_at IS NULL)';
        $depWherePlain[] = '(deleted_at IS NULL)';
      }
      if ($hasColumn($pdo, $depositTable, 'activo')) {
        $depWhere[] = '(d.activo IS NULL OR d.activo = 1)';
        $depWherePlain[] = '(activo IS NULL OR activo = 1)';
      }

      $depWhereSql = implode(' AND ', $depWhere);
      $depWherePlainSql = implode(' AND ', $depWherePlain);

      $posSelect = "
        SELECT
          d.id   AS deposito_id,
          d.code AS deposito_code,
          COUNT(p.id) AS posiciones_totales,
          COALESCE(SUM(COALESCE(p.capacidad_pallets,1)), 0) AS capacidad_total_pallets
      ";
      if ($hasM2Column) {
        $posSelect .= ",
          COALESCE(SUM(COALESCE(p.capacidad_pallets,1) * COALESCE(p.m2_por_pallet, 0)), 0) AS m2_totales
        ";
      }
      $posSelect .= "
        FROM {$depositTable} d
        {$posJoin}
  WHERE {$depWhereSql}
        GROUP BY d.id, d.code
        ORDER BY d.code
      ";

      $positionRows = $pdo->query($posSelect)->fetchAll(PDO::FETCH_ASSOC);
      $positionsByDepot = [];
      foreach ($positionRows as $row) {
        $depId = (int)($row['deposito_id'] ?? 0);
        if ($depId <= 0) {
          continue;
        }
        $positionsByDepot[$depId] = [
          'deposito_code'          => (string)($row['deposito_code'] ?? ''),
          'posiciones_totales'     => (int)($row['posiciones_totales'] ?? 0),
          'capacidad_total_pallets' => (int)($row['capacidad_total_pallets'] ?? 0),
          'm2_totales'             => $hasM2Column ? (float)($row['m2_totales'] ?? 0.0) : 0.0,
        ];
      }

      $palletTable = null;
      foreach (['wh_pallet', 'wh_pallets'] as $candidate) {
        if ($hasTable($pdo, $candidate)) {
          $palletTable = $candidate;
          break;
        }
      }

      $palletRows = [];
      $palletColumn = null;
      $depositColumn = null;
      if ($palletTable) {
        foreach (['posicion_id', 'pos_id', 'position_id'] as $candidate) {
          if ($hasColumn($pdo, $palletTable, $candidate)) {
            $palletColumn = $candidate;
            break;
          }
        }
        foreach (['deposito_id', 'deposit_id', 'depositoId'] as $candidate) {
          if ($hasColumn($pdo, $palletTable, $candidate)) {
            $depositColumn = $candidate;
            break;
          }
        }

        if ($palletColumn) {
          $posJoinForPallet = 'LEFT JOIN ' . $positionTable . ' pos ON pos.id = pal.' . $palletColumn;
          $palWhere = ['pal.' . $palletColumn . ' IS NOT NULL', 'pal.' . $palletColumn . ' <> 0'];
          if ($hasColumn($pdo, $palletTable, 'deleted_at')) {
            $palWhere[] = '(pal.deleted_at IS NULL)';
          }

          $depositExpr = $depositColumn
            ? "COALESCE(pos.deposito_id, pal.{$depositColumn})"
            : 'pos.deposito_id';

          $palSelect = "
            SELECT
              {$depositExpr} AS deposito_id,
              COUNT(*) AS pallets_actuales
          ";
          if ($hasM2Column) {
            $palSelect .= ",
              COALESCE(SUM(COALESCE(pos.m2_por_pallet, 0)), 0) AS m2_ocupados
            ";
          }
          $palSelect .= "
            FROM {$palletTable} pal
            {$posJoinForPallet}
            WHERE " . implode(' AND ', $palWhere) . "
            GROUP BY deposito_id
            HAVING deposito_id IS NOT NULL
          ";
          $palletRows = $pdo->query($palSelect)->fetchAll(PDO::FETCH_ASSOC);
        }
      }

      $palletsByDepot = [];
      foreach ($palletRows as $row) {
        $depId = (int)($row['deposito_id'] ?? 0);
        if ($depId <= 0) {
          continue;
        }
        $palletsByDepot[$depId] = [
          'pallets_actuales' => (int)($row['pallets_actuales'] ?? 0),
          'm2_ocupados'      => $hasM2Column ? (float)($row['m2_ocupados'] ?? 0.0) : 0.0,
        ];
      }

      $allDepotIds = array_values(array_unique(array_merge(array_keys($positionsByDepot), array_keys($palletsByDepot))));
      if (!$allDepotIds && $depositTable) {
        $stmt = $pdo->query('SELECT id FROM ' . $depositTable . ' WHERE ' . $depWherePlainSql);
        $allDepotIds = array_map('intval', $stmt->fetchAll(PDO::FETCH_COLUMN));
      }

      if ($allDepotIds) {
        $missing = array_diff($allDepotIds, array_keys($positionsByDepot));
        if ($missing) {
          $placeholders = implode(',', array_fill(0, count($missing), '?'));
          $stmt = $pdo->prepare('SELECT id, code FROM ' . $depositTable . ' WHERE id IN (' . $placeholders . ')');
          $stmt->execute(array_values($missing));
          while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $depId = (int)($row['id'] ?? 0);
            if ($depId <= 0) {
              continue;
            }
            $positionsByDepot[$depId] = [
              'deposito_code'          => (string)($row['code'] ?? ''),
              'posiciones_totales'     => 0,
              'capacidad_total_pallets' => 0,
              'm2_totales'             => 0.0,
            ];
          }
        }
      }

      sort($allDepotIds);
      foreach ($allDepotIds as $depId) {
        $positionInfo = $positionsByDepot[$depId] ?? [
          'deposito_code'          => '',
          'posiciones_totales'     => 0,
          'capacidad_total_pallets' => 0,
          'm2_totales'             => 0.0,
        ];
        $palletInfo = $palletsByDepot[$depId] ?? [
          'pallets_actuales' => 0,
          'm2_ocupados'      => 0.0,
        ];

        $capacidad   = max((int)$positionInfo['capacidad_total_pallets'], 0);
        $ocupados    = max((int)$palletInfo['pallets_actuales'], 0);
        $disponibles = max($capacidad - $ocupados, 0);

        $totalM2 = $hasM2Column
          ? (float)$positionInfo['m2_totales']
          : ($capacidad * $defaultM2);
        $m2Ocupados = $hasM2Column
          ? (float)$palletInfo['m2_ocupados']
          : ($ocupados * $defaultM2);
        $m2Disponibles = max($totalM2 - $m2Ocupados, 0.0);

        $porcentaje = $capacidad > 0
          ? round(min(($ocupados / $capacidad) * 100, 100), 1)
          : 0.0;

        $ocupacionDepositos[] = [
          'deposito_id'      => $depId,
          'deposito_code'    => $positionInfo['deposito_code'] ?: ('DEP-' . $depId),
          'capacidad'        => $capacidad,
          'ocupados'         => $ocupados,
          'disponibles'      => $disponibles,
          'porcentaje'       => $porcentaje,
          'm2_totales'       => $totalM2,
          'm2_ocupados'      => $m2Ocupados,
          'm2_disponibles'   => $m2Disponibles,
        ];

        $ocupacionTotales['capacidad_pallets']   += $capacidad;
        $ocupacionTotales['pallets_actuales']    += $ocupados;
        $ocupacionTotales['pallets_disponibles'] += $disponibles;
        $ocupacionTotales['m2_totales']          += $totalM2;
        $ocupacionTotales['m2_ocupados']         += $m2Ocupados;
        $ocupacionTotales['m2_disponibles']      += $m2Disponibles;
      }

      $ocupacionTotales['porcentaje'] = $ocupacionTotales['capacidad_pallets'] > 0
        ? round(min(($ocupacionTotales['pallets_actuales'] / $ocupacionTotales['capacidad_pallets']) * 100, 100), 1)
        : 0.0;
    } else {
      $ocupacionError = 'Faltan tablas de depósito o posiciones para calcular la ocupación.';
    }
  } catch (Throwable $e) {
    $ocupacionError = 'Error al calcular la ocupación del depósito.';
  }
} else {
  $ocupacionError = 'No se pudo establecer conexión con la base de datos.';
}
?>
<div class="row g-3 mt-1">
  <div class="col-12 col-lg-6">
    <div class="card h-100">
      <div class="card-header d-flex align-items-center justify-content-between">
        <h6 class="mb-0">Buscar stock actual de un producto</h6>
      </div>
      <div class="card-body">
        <form id="frmStockProducto" class="row g-2" onsubmit="return false;">
          <div class="col-12">
            <label for="sp_q" class="form-label">Producto (SKU o nombre)</label>
            <input type="text" class="form-control" id="sp_q" placeholder="Ej.: ABC-123 o 'Yerba'" />
          </div>
          <div class="col-7 col-md-8">
            <label for="sp_dep" class="form-label">Depósito (opcional)</label>
            <input type="text" class="form-control" id="sp_dep" placeholder="Ej.: DEP1" />
          </div>
          <div class="col-5 col-md-4 d-flex align-items-end">
            <button id="sp_btn" class="btn btn-primary w-100" type="button">Buscar</button>
          </div>
        </form>
        <div id="sp_msg" class="mt-2 small text-muted"></div>
        <div id="sp_results" class="mt-3" style="max-height: 360px; overflow:auto;"></div>

        <script>
          (function(){
            const $ = (sel) => document.querySelector(sel);
            const btn = $('#sp_btn');
            const qEl = $('#sp_q');
            const depEl = $('#sp_dep');
            const msg = $('#sp_msg');
            const out = $('#sp_results');

            function esc(s){ return (s||'').toString().replace(/[&<>"]/g, c=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;' }[c])); }

            async function doSearch(){
              const q = (qEl.value||'').trim();
              const deposito_code = (depEl.value||'').trim();
              if(!q){ msg.textContent = 'Indicá un SKU o nombre de producto.'; qEl.focus(); return; }
              msg.textContent = 'Buscando…';
              out.innerHTML = '';
              btn.disabled = true;
              try {
                const params = new URLSearchParams({ q });
                if(deposito_code) params.set('deposito_code', deposito_code);
                const res = await fetch('/api/inventario/stock_producto.php?' + params.toString(), { credentials: 'same-origin' });
                const js = await res.json();
                if(!js.ok){
                  if(js.matches && js.matches.length){
                    const lis = js.matches.map(m => `<li><a href="#" data-sku="${esc(m.sku)}" class="sp-pick">${esc(m.sku)} — ${esc(m.denominacion)}</a></li>`).join('');
                    out.innerHTML = `<div class="alert alert-warning">${esc(js.error||'Ambiguo: refiná la búsqueda.')}</div><ul>${lis}</ul>`;
                    out.querySelectorAll('a.sp-pick').forEach(a => a.addEventListener('click', (ev) => {
                      ev.preventDefault();
                      qEl.value = ev.currentTarget.getAttribute('data-sku')||'';
                      doSearch();
                    }));
                  } else {
                    out.innerHTML = `<div class="alert alert-danger">${esc(js.error||'Sin resultados')}</div>`;
                  }
                  msg.textContent = '';
                  return;
                }
                msg.textContent = `Producto: ${js.producto.sku} — ${js.producto.denominacion}${js.deposito? ' | Depósito: ' + js.deposito.code : ''}`;
                const rows = js.rows||[];
                if(rows.length===0){ out.innerHTML = '<div class="text-muted">Sin stock para este producto.</div>'; return; }
                const th = '<tr><th>Ambiente</th><th>Posición</th><th>Lote</th><th>Venc</th><th class="text-end">UV</th><th class="text-end">UC</th><th class="text-end">UV equiv (UC)</th></tr>';
                const trs = rows.map(r => `<tr>
                  <td>${esc(r.ambiente||'')}</td>
                  <td>${esc(r.pos_code||'')}</td>
                  <td>${esc(r.lote_codigo||'')}</td>
                  <td>${esc(r.venc||'')}</td>
                  <td class="text-end">${r.qty_uv||0}</td>
                  <td class="text-end">${r.qty_uc||0}</td>
                  <td class="text-end">${r.uv_equiv_from_uc||0}</td>
                </tr>`).join('');
                const tot = js.totals||{};
                const tfoot = `<tr class="fw-bold"><td colspan="4" class="text-end">Totales</td><td class="text-end">${tot.total_uv||0}</td><td class="text-end">${tot.total_uc||0}</td><td class="text-end">${tot.total_uv_equiv_from_uc||0}</td></tr>`;
                out.innerHTML = `<div class="table-responsive"><table class="table table-sm table-striped align-middle"><thead>${th}</thead><tbody>${trs}</tbody><tfoot>${tfoot}</tfoot></table></div>`;
              } catch (e){
                out.innerHTML = `<div class="alert alert-danger">Error consultando el API</div>`;
              } finally {
                btn.disabled = false;
              }
            }

            btn.addEventListener('click', doSearch);
            qEl.addEventListener('keydown', (ev)=>{ if(ev.key==='Enter'){ ev.preventDefault(); doSearch(); } });
          })();
        </script>
      </div>
    </div>
  </div>
  <div class="col-12 col-lg-6">
    <div class="card h-100">
      <div class="card-header"><h6 class="mb-0">Ocupación por Depósito</h6></div>
      <div class="card-body">
        <?php if ($ocupacionError): ?>
          <div class="alert alert-warning mb-0 small"><?= htmlspecialchars($ocupacionError, ENT_QUOTES, 'UTF-8') ?></div>
        <?php elseif (!$ocupacionDepositos): ?>
          <div class="text-dark">No hay información de ocupación disponible.</div>
        <?php else: ?>
          <div class="row g-2">
            <div class="col-12 col-md-6">
              <div class="border rounded p-3 bg-secondary h-100">
                <div class="small text-uppercase text-dark mb-1">Pallets</div>
                <div class="fs-5 fw-semibold mb-1 text-dark">
                  <?= number_format($ocupacionTotales['pallets_actuales'], 0, ',', '.') ?>
                  <span class="fw-normal text-dark">ocupados</span>
                </div>
                <div class="small text-dark">
                  Capacidad <?= number_format($ocupacionTotales['capacidad_pallets'], 0, ',', '.') ?> ·
                  Disponibles <?= number_format($ocupacionTotales['pallets_disponibles'], 0, ',', '.') ?> ·
                  <?= number_format($ocupacionTotales['porcentaje'], 1, ',', '.') ?>%
                </div>
              </div>
            </div>
            <div class="col-12 col-md-6">
              <div class="border rounded p-3 bg-primary h-100">
                <div class="small text-uppercase text-white mb-1">Superficie</div>
                <div class="fs-5 fw-semibold mb-1 text-white">
                  <?= number_format($ocupacionTotales['m2_ocupados'], 1, ',', '.') ?> m2
                  <span class="fw-normal text-white">ocupados</span>
                </div>
                <div class="small text-white">
                  Total <?= number_format($ocupacionTotales['m2_totales'], 1, ',', '.') ?> m2 ·
                  Disponibles <?= number_format($ocupacionTotales['m2_disponibles'], 1, ',', '.') ?> m2
                </div>
              </div>
            </div>
          </div>

          <div class="table-responsive mt-3">
            <table class="table table-sm align-middle mb-0">
              <thead class="table-dark">
                <tr>
                  <th>Depósito</th>
                  <th class="text-end">Pallets ocup.</th>
                  <th class="text-end">Pallets disp.</th>
                  <th class="text-end">% Ocup.</th>
                  <th class="text-end">m2 ocup.</th>
                  <th class="text-end">m2 disp.</th>
                </tr>
              </thead>
              <tbody>
                <?php foreach ($ocupacionDepositos as $dep): ?>
                  <tr>
                    <td><?= htmlspecialchars($dep['deposito_code'], ENT_QUOTES, 'UTF-8') ?></td>
                    <td class="text-end"><?= number_format($dep['ocupados'], 0, ',', '.') ?></td>
                    <td class="text-end"><?= number_format($dep['disponibles'], 0, ',', '.') ?></td>
                    <td class="text-end"><?= number_format($dep['porcentaje'], 1, ',', '.') ?>%</td>
                    <td class="text-end"><?= number_format($dep['m2_ocupados'], 1, ',', '.') ?></td>
                    <td class="text-end"><?= number_format($dep['m2_disponibles'], 1, ',', '.') ?></td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>

          <?php if ($ocupacionNota): ?>
            <p class="small text-muted mt-2 mb-0"><?= htmlspecialchars($ocupacionNota, ENT_QUOTES, 'UTF-8') ?></p>
          <?php endif; ?>
        <?php endif; ?>
      </div>
    </div>
  </div>
</div>
