<?php
declare(strict_types=1);
/**
 * Sección: Tarjetas de métricas (datos reales)
 * - Usa helpers (project_path) para cargar DB
 * - KPIs calculados con PDO y fallbacks seguros
 */

// Helpers/DB
if (!function_exists('project_path')) {
  function project_path(string $rel): string {
    return rtrim(dirname(__DIR__, 4), DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . ltrim($rel, DIRECTORY_SEPARATOR);
  }
}
require_once project_path('config/db.php');

$pdo = null;
try { $pdo = getPDO(); } catch (Throwable $e) { /* silencioso en home */ }

// Valores por defecto
$palletsUbicados     = 0;
$palletsPicking      = 0;
$movimientosHoy      = 0;
$recepcionesPend     = 0;
$ordenesSalidaHoy    = 0;
$palletsCuarentena   = 0;
$totalPesoKg         = 0.0;
$totalValorGs        = 0.0;
$totalCajasCompletas = 0;
$totalUcSueltas      = 0;
$ucSueltasEqCajas    = 0.0;
// Desglose de valor/peso: Disponible vs Cuarentena
$pesoDispKg          = 0.0;
$valorDispGs         = 0.0;
$pesoCuarKg          = 0.0;
$valorCuarGs         = 0.0;
// Desglose de cajas/UC: Disponible vs Cuarentena
$cajasDisp           = 0;
$ucDisp              = 0;
$eqCajasDisp         = 0.0;
$cajasCuar           = 0;
$ucCuar              = 0;
$eqCajasCuar         = 0.0;

// Fecha de trabajo (hoy)
$hoy = date('Y-m-d');

if ($pdo instanceof PDO) {
  try {
    // ===== Pallets en PICKING =====
    // Detección de tabla/columna de posición (posicion_id | pos_id) y conteo por ambiente 'PICKING'
    $hasCol = function(PDO $pdo, string $tbl, string $col): bool {
      $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?");
      $st->execute([$tbl, $col]);
      return (int)$st->fetchColumn() > 0;
    };
    $hasTbl = function(PDO $pdo, string $t): bool {
      static $cache = [];
      if (array_key_exists($t, $cache)) return $cache[$t];
      $st = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
      $st->execute([$t]);
      return $cache[$t] = ((int)$st->fetchColumn() > 0);
    };

    $palTbl = $hasTbl($pdo, 'wh_pallet') ? 'wh_pallet' : ($hasTbl($pdo, 'wh_pallets') ? 'wh_pallets' : null);
    $posCol = null;
    if ($palTbl) {
      if ($hasCol($pdo, $palTbl, 'posicion_id'))      $posCol = 'posicion_id';
      elseif ($hasCol($pdo, $palTbl, 'pos_id'))       $posCol = 'pos_id';
    }
    if ($palTbl && $posCol) {
      // Excluir EMBARCADO si existe el estado en legacy
      $embId = 0;
      try { $embId = (int)($pdo->query("SELECT id FROM wh_pallet_estado WHERE code='EMBARCADO' LIMIT 1")->fetchColumn() ?: 0); } catch (Throwable $e) {}

      $sqlPicking = "
        SELECT COUNT(*)
          FROM {$palTbl} p
          JOIN wh_posicion pos ON pos.id = p.{$posCol}
          JOIN wh_ambiente a  ON a.id = pos.ambiente_id AND a.code='PICKING'
         WHERE p.{$posCol} IS NOT NULL
      ";
      if ($embId && $palTbl === 'wh_pallet') { $sqlPicking .= " AND (p.estado_id IS NULL OR p.estado_id <> {$embId})"; }
      $palletsPicking = (int)($pdo->query($sqlPicking)->fetchColumn() ?: 0);
    }

    // ===== Peso y Valor del depósito (disponible vs cuarentena) =====
    // Estrategia:
    // 1) Preferir wh_stock (qty_uc actual) x para_producto_pack (peso/precio por UC)
    // 2) Si no existe wh_stock, usar wh_pallet_items (nuevo esquema) con uc_total_cache
    // 3) Si sólo existe wh_pallet_item (legacy), usar uc_unidades (subestima cajas si no hay uc_por_caja)
    // (usa el mismo helper $hasTbl definido arriba)
  $hasStock = $hasTbl($pdo, 'wh_stock');
  $hasPiNew = $hasTbl($pdo, 'wh_pallet_items');
  $hasPiOld = $hasTbl($pdo, 'wh_pallet_item');

    // Derived table picking one pack row per producto (max as pragmatic default)
    $packAgg = "(
      SELECT producto_id,
             MAX(precio_uc_efectivo_gs) AS precio_uc_efectivo_gs,
             MAX(peso_estandar_kg)      AS peso_estandar_kg
        FROM para_producto_pack
       GROUP BY producto_id
    )";
    // UC por caja por producto (para equivalencias)
    $packUnitsAgg = "(
      SELECT producto_id,
             MAX(unidades_por_uv) AS unidades_por_uv
        FROM para_producto_pack
       GROUP BY producto_id
    )";

    if ($hasStock) {
      // Disponible/Cuarentena por estado del PALLET (fuente de verdad); fallback a ambiente si no hay estado
      $embId = 0; $cuaId = 0;
      try { $embId = (int)($pdo->query("SELECT id FROM wh_pallet_estado WHERE code='EMBARCADO' LIMIT 1")->fetchColumn() ?: 0); } catch (Throwable $e) {}
      try { $cuaId = (int)($pdo->query("SELECT id FROM wh_pallet_estado WHERE code='CUARENTENA' LIMIT 1")->fetchColumn() ?: 0); } catch (Throwable $e) {}

      // uc_expresion: convierte UV (cajas) a UC usando pack, suma unidades sueltas
  $ucExprStock = "(GREATEST(COALESCE(s.qty_uc,0),0) + GREATEST(COALESCE(s.qty_uv,0),0) * COALESCE(pu.unidades_por_uv, 0))";

      if ($cuaId) {
        $sqlDisp = "
          SELECT
            COALESCE(SUM({$ucExprStock} * COALESCE(pp.peso_estandar_kg, 0)), 0) AS kg,
            COALESCE(SUM({$ucExprStock} * COALESCE(pp.precio_uc_efectivo_gs, 0)), 0) AS gs
          FROM wh_stock s
          LEFT JOIN wh_pallet p ON p.id = s.pallet_id
          LEFT JOIN {$packAgg} pp ON pp.producto_id = s.producto_id
          LEFT JOIN {$packUnitsAgg} pu ON pu.producto_id = s.producto_id
          WHERE (s.qty_uc > 0 OR s.qty_uv > 0)
            " . ($embId ? " AND (p.estado_id IS NULL OR p.estado_id <> {$embId})" : "") . "
            AND (p.estado_id IS NULL OR p.estado_id <> {$cuaId})
        ";
        $rowD = $pdo->query($sqlDisp)->fetch(PDO::FETCH_ASSOC) ?: [];
        $pesoDispKg  = (float)($rowD['kg'] ?? 0);
        $valorDispGs = (float)($rowD['gs'] ?? 0);

        $sqlCua = "
          SELECT
            COALESCE(SUM({$ucExprStock} * COALESCE(pp.peso_estandar_kg, 0)), 0) AS kg,
            COALESCE(SUM({$ucExprStock} * COALESCE(pp.precio_uc_efectivo_gs, 0)), 0) AS gs
          FROM wh_stock s
          JOIN wh_pallet p ON p.id = s.pallet_id AND p.estado_id = {$cuaId}
          LEFT JOIN {$packAgg} pp ON pp.producto_id = s.producto_id
          LEFT JOIN {$packUnitsAgg} pu ON pu.producto_id = s.producto_id
        ";
        $rowC = $pdo->query($sqlCua)->fetch(PDO::FETCH_ASSOC) ?: [];
        $pesoCuarKg  = (float)($rowC['kg'] ?? 0);
        $valorCuarGs = (float)($rowC['gs'] ?? 0);
      } else {
        // Fallback por ambiente si no hay estado de pallets
        $sqlDisp = "
          SELECT
            COALESCE(SUM({$ucExprStock} * COALESCE(pp.peso_estandar_kg, 0)), 0) AS kg,
            COALESCE(SUM({$ucExprStock} * COALESCE(pp.precio_uc_efectivo_gs, 0)), 0) AS gs
          FROM wh_stock s
          LEFT JOIN wh_posicion pos ON pos.id = s.posicion_id
          LEFT JOIN wh_ambiente amb ON amb.id = pos.ambiente_id
          LEFT JOIN {$packAgg} pp ON pp.producto_id = s.producto_id
          LEFT JOIN {$packUnitsAgg} pu ON pu.producto_id = s.producto_id
          WHERE s.posicion_id IS NOT NULL
            AND (amb.code IS NULL OR amb.code <> 'CUARENTENA')
        ";
        $rowD = $pdo->query($sqlDisp)->fetch(PDO::FETCH_ASSOC) ?: [];
        $pesoDispKg  = (float)($rowD['kg'] ?? 0);
        $valorDispGs = (float)($rowD['gs'] ?? 0);

        $sqlCua = "
          SELECT
            COALESCE(SUM({$ucExprStock} * COALESCE(pp.peso_estandar_kg, 0)), 0) AS kg,
            COALESCE(SUM({$ucExprStock} * COALESCE(pp.precio_uc_efectivo_gs, 0)), 0) AS gs
          FROM wh_stock s
          JOIN wh_posicion pos ON pos.id = s.posicion_id
          JOIN wh_ambiente amb ON amb.id = pos.ambiente_id AND amb.code = 'CUARENTENA'
          LEFT JOIN {$packAgg} pp ON pp.producto_id = s.producto_id
          LEFT JOIN {$packUnitsAgg} pu ON pu.producto_id = s.producto_id
        ";
        $rowC = $pdo->query($sqlCua)->fetch(PDO::FETCH_ASSOC) ?: [];
        $pesoCuarKg  = (float)($rowC['kg'] ?? 0);
        $valorCuarGs = (float)($rowC['gs'] ?? 0);
      }
    } elseif ($hasPiNew) {
      // Nuevo core (tabla wh_pallet_items): usar columnas uv_cajas / uc_unidades y estado desde wh_pallet(es)
      $palTblJoin = $hasTbl($pdo, 'wh_pallet') ? 'wh_pallet' : ($hasTbl($pdo, 'wh_pallets') ? 'wh_pallets' : null);
      $hasPalDeleted = $palTblJoin && $hasCol($pdo, $palTblJoin, 'deleted_at');
      $embId = 0; $cuaId = 0;
      try { $embId = (int)($pdo->query("SELECT id FROM wh_pallet_estado WHERE code='EMBARCADO' LIMIT 1")->fetchColumn() ?: 0); } catch (Throwable $e) {}
      try { $cuaId = (int)($pdo->query("SELECT id FROM wh_pallet_estado WHERE code='CUARENTENA' LIMIT 1")->fetchColumn() ?: 0); } catch (Throwable $e) {}

  $ucExprItems = "(COALESCE(it.uv_cajas,0) * COALESCE(pu.unidades_por_uv, 0) + COALESCE(it.uc_unidades,0))";
      $baseJoin = "
        FROM wh_pallet_items it
        " . ($palTblJoin ? "JOIN {$palTblJoin} p ON p.id = it.pallet_id" : '') . "
        LEFT JOIN {$packAgg} pp ON pp.producto_id = it.producto_id
        LEFT JOIN {$packUnitsAgg} pu ON pu.producto_id = it.producto_id
      ";

      $dispFilters = '';
      if ($palTblJoin) {
        if ($hasPalDeleted) { $dispFilters .= " AND (p.deleted_at IS NULL)"; }
        if ($embId) { $dispFilters .= " AND (p.estado_id IS NULL OR p.estado_id <> {$embId})"; }
        if ($cuaId) { $dispFilters .= " AND (p.estado_id IS NULL OR p.estado_id <> {$cuaId})"; }
      }

      $sqlD = "
        SELECT
          COALESCE(SUM({$ucExprItems} * COALESCE(pp.peso_estandar_kg, 0)), 0) AS kg,
          COALESCE(SUM({$ucExprItems} * COALESCE(pp.precio_uc_efectivo_gs, 0)), 0) AS gs
        {$baseJoin}
        WHERE 1 = 1{$dispFilters}
      ";
      $rowD = $pdo->query($sqlD)->fetch(PDO::FETCH_ASSOC) ?: [];
      $pesoDispKg  = (float)($rowD['kg'] ?? 0);
      $valorDispGs = (float)($rowD['gs'] ?? 0);

      if ($palTblJoin && $cuaId) {
        $cuaFilters = '';
        if ($hasPalDeleted) { $cuaFilters .= " AND (p.deleted_at IS NULL)"; }
        $cuaFilters .= " AND p.estado_id = {$cuaId}";
        $sqlC = "
          SELECT
            COALESCE(SUM({$ucExprItems} * COALESCE(pp.peso_estandar_kg, 0)), 0) AS kg,
            COALESCE(SUM({$ucExprItems} * COALESCE(pp.precio_uc_efectivo_gs, 0)), 0) AS gs
          {$baseJoin}
          WHERE 1 = 1{$cuaFilters}
        ";
        $rowC = $pdo->query($sqlC)->fetch(PDO::FETCH_ASSOC) ?: [];
        $pesoCuarKg  = (float)($rowC['kg'] ?? 0);
        $valorCuarGs = (float)($rowC['gs'] ?? 0);
      }

      // Totales de cajas y UC sueltas + equivalencia a cajas (sumatoria UC/unidad_por_uc)
      $sqlCnt = "
        SELECT
          COALESCE(SUM(COALESCE(it.uv_cajas,0)), 0) AS cajas_total,
          COALESCE(SUM(COALESCE(it.uc_unidades,0)), 0) AS uc_unidades_total,
          COALESCE(SUM(COALESCE(it.uc_unidades,0) / NULLIF(COALESCE(pu.unidades_por_uv,0),0)), 0) AS eq_cajas
        FROM wh_pallet_items it
        LEFT JOIN {$packUnitsAgg} pu ON pu.producto_id = it.producto_id
      ";
      $rCnt = $pdo->query($sqlCnt)->fetch(PDO::FETCH_ASSOC) ?: [];
      $totalCajasCompletas = (int)($rCnt['cajas_total'] ?? 0);
      $totalUcSueltas      = (int)($rCnt['uc_unidades_total'] ?? 0);
      $ucSueltasEqCajas    = (float)($rCnt['eq_cajas'] ?? 0);
    } elseif ($hasPiOld) {
      // Legacy muy básico: sólo uc_unidades (no conocemos uc_por_caja aquí)
      // Disponible/Cuarentena por estado del PALLET; fallback a ambiente si no hay estado
      $embId = 0; $cuaId = 0;
      try { $embId = (int)($pdo->query("SELECT id FROM wh_pallet_estado WHERE code='EMBARCADO' LIMIT 1")->fetchColumn() ?: 0); } catch (Throwable $e) {}
      try { $cuaId = (int)($pdo->query("SELECT id FROM wh_pallet_estado WHERE code='CUARENTENA' LIMIT 1")->fetchColumn() ?: 0); } catch (Throwable $e) {}

      if ($cuaId) {
        $sqlD = "
          SELECT
            COALESCE(SUM(COALESCE(pi.uc_unidades,0) * COALESCE(pp.peso_estandar_kg, 0)), 0) AS kg,
            COALESCE(SUM(COALESCE(pi.uc_unidades,0) * COALESCE(pp.precio_uc_efectivo_gs, 0)), 0) AS gs
          FROM wh_pallet_item pi
          JOIN wh_pallet p ON p.id = pi.pallet_id
          LEFT JOIN {$packAgg} pp ON pp.producto_id = pi.producto_id
          WHERE (p.posicion_id IS NOT NULL)
            " . ($embId ? " AND (p.estado_id IS NULL OR p.estado_id <> {$embId})" : "") . "
            AND (p.estado_id IS NULL OR p.estado_id <> {$cuaId})
        ";
        $rowD = $pdo->query($sqlD)->fetch(PDO::FETCH_ASSOC) ?: [];
        $pesoDispKg  = (float)($rowD['kg'] ?? 0);
        $valorDispGs = (float)($rowD['gs'] ?? 0);

        $sqlC = "
          SELECT
            COALESCE(SUM(COALESCE(pi.uc_unidades,0) * COALESCE(pp.peso_estandar_kg, 0)), 0) AS kg,
            COALESCE(SUM(COALESCE(pi.uc_unidades,0) * COALESCE(pp.precio_uc_efectivo_gs, 0)), 0) AS gs
          FROM wh_pallet_item pi
          JOIN wh_pallet p ON p.id = pi.pallet_id AND p.estado_id = {$cuaId}
          LEFT JOIN {$packAgg} pp ON pp.producto_id = pi.producto_id
        ";
        $rowC = $pdo->query($sqlC)->fetch(PDO::FETCH_ASSOC) ?: [];
        $pesoCuarKg  = (float)($rowC['kg'] ?? 0);
        $valorCuarGs = (float)($rowC['gs'] ?? 0);
      } else {
        // Fallback por ambiente
        $sqlD = "
          SELECT
            COALESCE(SUM(COALESCE(pi.uc_unidades,0) * COALESCE(pp.peso_estandar_kg, 0)), 0) AS kg,
            COALESCE(SUM(COALESCE(pi.uc_unidades,0) * COALESCE(pp.precio_uc_efectivo_gs, 0)), 0) AS gs
          FROM wh_pallet_item pi
          JOIN wh_pallet p ON p.id = pi.pallet_id
          LEFT JOIN wh_posicion pos ON pos.id = p.posicion_id
          LEFT JOIN wh_ambiente amb ON amb.id = pos.ambiente_id
          LEFT JOIN {$packAgg} pp ON pp.producto_id = pi.producto_id
          WHERE (p.posicion_id IS NOT NULL)
            AND (amb.code IS NULL OR amb.code <> 'CUARENTENA')
        ";
        $rowD = $pdo->query($sqlD)->fetch(PDO::FETCH_ASSOC) ?: [];
        $pesoDispKg  = (float)($rowD['kg'] ?? 0);
        $valorDispGs = (float)($rowD['gs'] ?? 0);

        $sqlC = "
          SELECT
            COALESCE(SUM(COALESCE(pi.uc_unidades,0) * COALESCE(pp.peso_estandar_kg, 0)), 0) AS kg,
            COALESCE(SUM(COALESCE(pi.uc_unidades,0) * COALESCE(pp.precio_uc_efectivo_gs, 0)), 0) AS gs
          FROM wh_pallet_item pi
          JOIN wh_pallet p ON p.id = pi.pallet_id
          JOIN wh_posicion pos ON pos.id = p.posicion_id
          JOIN wh_ambiente amb ON amb.id = pos.ambiente_id AND amb.code = 'CUARENTENA'
          LEFT JOIN {$packAgg} pp ON pp.producto_id = pi.producto_id
        ";
        $rowC = $pdo->query($sqlC)->fetch(PDO::FETCH_ASSOC) ?: [];
        $pesoCuarKg  = (float)($rowC['kg'] ?? 0);
        $valorCuarGs = (float)($rowC['gs'] ?? 0);
      }

      // Cajas y UC sueltas desde legacy
      $row2 = $pdo->query("SELECT COALESCE(SUM(uv_cajas),0) AS cajas_total, COALESCE(SUM(uc_unidades),0) AS uc_sueltas_total FROM wh_pallet_item")
                  ->fetch(PDO::FETCH_ASSOC) ?: [];
      $totalCajasCompletas = (int)($row2['cajas_total'] ?? 0);
      $totalUcSueltas      = (int)($row2['uc_sueltas_total'] ?? 0);
      $ucSueltasEqCajas    = 0.0; // No conocemos uc_por_caja aquí
    }
  } catch (Throwable $e) {
    // Silencioso: si algo falla, dejar 0 y seguir mostrando el resto de KPIs
  }

  // Recalcular Cajas/UC sueltas con detección de tabla con datos (evitar 0s si la tabla nueva existe pero está vacía)
  try {
    $rowCount = function(PDO $pdo, string $tbl): int {
      $st = $pdo->prepare("SELECT COUNT(*) FROM `{$tbl}`");
      $st->execute();
      return (int)$st->fetchColumn();
    };

    $rowsNew = $hasPiNew ? $rowCount($pdo, 'wh_pallet_items') : 0;
    $rowsOld = $hasPiOld ? $rowCount($pdo, 'wh_pallet_item') : 0;

    if ($rowsNew > 0) {
      $sqlCnt = "
        SELECT
          COALESCE(SUM(COALESCE(it.uv_cajas,0)), 0) AS cajas_total,
          COALESCE(SUM(COALESCE(it.uc_unidades,0)), 0) AS uc_unidades_total,
          COALESCE(SUM(COALESCE(it.uc_unidades,0) / NULLIF(COALESCE(pu.unidades_por_uv,0), 0)), 0) AS eq_cajas
        FROM wh_pallet_items it
        LEFT JOIN {$packUnitsAgg} pu ON pu.producto_id = it.producto_id
      ";
      $rCnt = $pdo->query($sqlCnt)->fetch(PDO::FETCH_ASSOC) ?: [];
      $totalCajasCompletas = (int)($rCnt['cajas_total'] ?? 0);
      $totalUcSueltas      = (int)($rCnt['uc_unidades_total'] ?? 0);
      $ucSueltasEqCajas    = (float)($rCnt['eq_cajas'] ?? 0);
    } elseif ($rowsOld > 0) {
      $sqlCnt = "
        SELECT
          COALESCE(SUM(COALESCE(it.uv_cajas,0)), 0) AS cajas_total,
          COALESCE(SUM(COALESCE(it.uc_unidades,0)), 0) AS uc_sueltas_total
        FROM wh_pallet_item it
      ";
      $rCnt = $pdo->query($sqlCnt)->fetch(PDO::FETCH_ASSOC) ?: [];
      $totalCajasCompletas = (int)($rCnt['cajas_total'] ?? 0);
      $totalUcSueltas      = (int)($rCnt['uc_sueltas_total'] ?? 0);
      $ucSueltasEqCajas    = 0.0; // no conocemos uc_por_caja
    } elseif ($hasStock) {
      // Fallback: usar wh_stock si no hay granularidad de items
      $sqlCnt = "SELECT COALESCE(SUM(qty_uv),0) AS cajas_total, COALESCE(SUM(qty_uc),0) AS uc_total FROM wh_stock";
      $rCnt = $pdo->query($sqlCnt)->fetch(PDO::FETCH_ASSOC) ?: [];
      $totalCajasCompletas = (int)($rCnt['cajas_total'] ?? 0);
      $totalUcSueltas      = (int)($rCnt['uc_total'] ?? 0);
      $ucSueltasEqCajas    = 0.0;
    }
  } catch (Throwable $e) {
    // no-op
  }

  // Desglose Disponible vs Cuarentena para cajas/UC
  try {
    // EMBARCADO y CUARENTENA id (legacy) para partir por estado del pallet
    $embId = 0; $cuaId = 0;
    try { $embId = (int)($pdo->query("SELECT id FROM wh_pallet_estado WHERE code='EMBARCADO' LIMIT 1")->fetchColumn() ?: 0); } catch (Throwable $e) {}
    try { $cuaId = (int)($pdo->query("SELECT id FROM wh_pallet_estado WHERE code='CUARENTENA' LIMIT 1")->fetchColumn() ?: 0); } catch (Throwable $e) {}

    if (isset($rowsNew) && $rowsNew > 0) {
      $palTblJoin = $hasTbl($pdo, 'wh_pallet') ? 'wh_pallet' : ($hasTbl($pdo, 'wh_pallets') ? 'wh_pallets' : null);
      $hasPalDeleted = $palTblJoin && $hasCol($pdo, $palTblJoin, 'deleted_at');
      $unitJoin = "LEFT JOIN {$packUnitsAgg} pu ON pu.producto_id = it.producto_id";

      $dispWhere = '';
      if ($palTblJoin) {
        if ($hasPalDeleted) { $dispWhere .= " AND (p.deleted_at IS NULL)"; }
        if ($embId) { $dispWhere .= " AND (p.estado_id IS NULL OR p.estado_id <> {$embId})"; }
        if ($cuaId) { $dispWhere .= " AND (p.estado_id IS NULL OR p.estado_id <> {$cuaId})"; }
      }

      $sqlDisp = "
        SELECT
          COALESCE(SUM(COALESCE(it.uv_cajas,0)), 0) AS cajas,
          COALESCE(SUM(COALESCE(it.uc_unidades,0)), 0) AS uc_unidades,
          COALESCE(SUM( COALESCE(it.uc_unidades,0) / NULLIF(COALESCE(pu.unidades_por_uv, 0), 0) ), 0) AS eq_cajas
        FROM wh_pallet_items it
        " . ($palTblJoin ? "JOIN {$palTblJoin} p ON p.id = it.pallet_id" : '') . "
        {$unitJoin}
        WHERE 1 = 1{$dispWhere}
      ";
      $d = $pdo->query($sqlDisp)->fetch(PDO::FETCH_ASSOC) ?: [];
      $cajasDisp   = (int)($d['cajas'] ?? 0);
      $ucDisp      = (int)($d['uc_unidades'] ?? 0);
      $eqCajasDisp = (float)($d['eq_cajas'] ?? 0);

      if ($palTblJoin && $cuaId) {
        $cuaWhere = '';
        if ($hasPalDeleted) { $cuaWhere .= " AND (p.deleted_at IS NULL)"; }
        $cuaWhere .= " AND p.estado_id = {$cuaId}";
        $sqlCua = "
          SELECT
            COALESCE(SUM(COALESCE(it.uv_cajas,0)), 0) AS cajas,
            COALESCE(SUM(COALESCE(it.uc_unidades,0)), 0) AS uc_unidades,
            COALESCE(SUM( COALESCE(it.uc_unidades,0) / NULLIF(COALESCE(pu.unidades_por_uv, 0), 0) ), 0) AS eq_cajas
          FROM wh_pallet_items it
          JOIN {$palTblJoin} p ON p.id = it.pallet_id
          {$unitJoin}
          WHERE 1 = 1{$cuaWhere}
        ";
        $c = $pdo->query($sqlCua)->fetch(PDO::FETCH_ASSOC) ?: [];
        $cajasCuar   = (int)($c['cajas'] ?? 0);
        $ucCuar      = (int)($c['uc_unidades'] ?? 0);
        $eqCajasCuar = (float)($c['eq_cajas'] ?? 0);
      }
    } elseif (isset($rowsOld) && $rowsOld > 0) {
      // Legacy: por estado del PALLET cuando exista; fallback a ambiente si no hay estado
      if ($cuaId) {
        $sqlDisp = "
          SELECT
            COALESCE(SUM(COALESCE(it.uv_cajas,0)), 0) AS cajas,
            COALESCE(SUM(COALESCE(it.uc_unidades,0)), 0) AS uc,
            COALESCE(SUM( COALESCE(it.uc_unidades,0) / NULLIF(COALESCE(pu.unidades_por_uv, 0), 0) ), 0) AS eq_cajas
          FROM wh_pallet_item it
          JOIN wh_pallet p ON p.id = it.pallet_id
          LEFT JOIN {$packUnitsAgg} pu ON pu.producto_id = it.producto_id
          WHERE (p.posicion_id IS NOT NULL)
            " . ($embId ? " AND (p.estado_id IS NULL OR p.estado_id <> {$embId})" : "") . "
            AND (p.estado_id IS NULL OR p.estado_id <> {$cuaId})
        ";
        $d = $pdo->query($sqlDisp)->fetch(PDO::FETCH_ASSOC) ?: [];
        $cajasDisp   = (int)($d['cajas'] ?? 0);
        $ucDisp      = (int)($d['uc'] ?? 0);
        $eqCajasDisp = (float)($d['eq_cajas'] ?? 0);

        $sqlCua = "
          SELECT
            COALESCE(SUM(COALESCE(it.uv_cajas,0)), 0) AS cajas,
            COALESCE(SUM(COALESCE(it.uc_unidades,0)), 0) AS uc,
            COALESCE(SUM( COALESCE(it.uc_unidades,0) / NULLIF(COALESCE(pu.unidades_por_uv, 0), 0) ), 0) AS eq_cajas
          FROM wh_pallet_item it
          JOIN wh_pallet p ON p.id = it.pallet_id AND p.estado_id = {$cuaId}
          LEFT JOIN {$packUnitsAgg} pu ON pu.producto_id = it.producto_id
        ";
        $c = $pdo->query($sqlCua)->fetch(PDO::FETCH_ASSOC) ?: [];
        $cajasCuar   = (int)($c['cajas'] ?? 0);
        $ucCuar      = (int)($c['uc'] ?? 0);
        $eqCajasCuar = (float)($c['eq_cajas'] ?? 0);
      } else {
        $sqlDisp = "
          SELECT
            COALESCE(SUM(COALESCE(it.uv_cajas,0)), 0) AS cajas,
            COALESCE(SUM(COALESCE(it.uc_unidades,0)), 0) AS uc,
            COALESCE(SUM( COALESCE(it.uc_unidades,0) / NULLIF(COALESCE(pu.unidades_por_uv, 0), 0) ), 0) AS eq_cajas
          FROM wh_pallet_item it
          JOIN wh_pallet p ON p.id = it.pallet_id
          LEFT JOIN wh_posicion pos ON pos.id = p.posicion_id
          LEFT JOIN wh_ambiente amb ON amb.id = pos.ambiente_id
          LEFT JOIN {$packUnitsAgg} pu ON pu.producto_id = it.producto_id
          WHERE (p.posicion_id IS NOT NULL)
            AND (amb.code IS NULL OR amb.code <> 'CUARENTENA')
        ";
        $d = $pdo->query($sqlDisp)->fetch(PDO::FETCH_ASSOC) ?: [];
        $cajasDisp   = (int)($d['cajas'] ?? 0);
        $ucDisp      = (int)($d['uc'] ?? 0);
        $eqCajasDisp = (float)($d['eq_cajas'] ?? 0);

        $sqlCua = "
          SELECT
            COALESCE(SUM(COALESCE(it.uv_cajas,0)), 0) AS cajas,
            COALESCE(SUM(COALESCE(it.uc_unidades,0)), 0) AS uc,
            COALESCE(SUM( COALESCE(it.uc_unidades,0) / NULLIF(COALESCE(pu.unidades_por_uv, 0), 0) ), 0) AS eq_cajas
          FROM wh_pallet_item it
          JOIN wh_pallet p ON p.id = it.pallet_id
          JOIN wh_posicion pos ON pos.id = p.posicion_id
          JOIN wh_ambiente amb ON amb.id = pos.ambiente_id AND amb.code='CUARENTENA'
          LEFT JOIN {$packUnitsAgg} pu ON pu.producto_id = it.producto_id
        ";
        $c = $pdo->query($sqlCua)->fetch(PDO::FETCH_ASSOC) ?: [];
        $cajasCuar   = (int)($c['cajas'] ?? 0);
        $ucCuar      = (int)($c['uc'] ?? 0);
        $eqCajasCuar = (float)($c['eq_cajas'] ?? 0);
      }
    } elseif ($hasStock) {
      // Stock resumido por estado del pallet si existe; fallback a ambiente
      if ($cuaId) {
        $sqlDisp = "
          SELECT COALESCE(SUM(s.qty_uv),0) AS cajas, COALESCE(SUM(s.qty_uc),0) AS uc,
                 COALESCE(SUM( s.qty_uc / NULLIF(COALESCE(pu.unidades_por_uv, 0), 0) ), 0) AS eq_cajas
          FROM wh_stock s
          LEFT JOIN wh_pallet p ON p.id = s.pallet_id
          LEFT JOIN {$packUnitsAgg} pu ON pu.producto_id = s.producto_id
          WHERE (s.qty_uc > 0 OR s.qty_uv > 0)
            AND (p.estado_id IS NULL OR p.estado_id <> {$cuaId})
            " . ($embId ? " AND (p.estado_id IS NULL OR p.estado_id <> {$embId})" : "") . "
        ";
        $d = $pdo->query($sqlDisp)->fetch(PDO::FETCH_ASSOC) ?: [];
        $cajasDisp   = (int)($d['cajas'] ?? 0);
        $ucDisp      = (int)($d['uc'] ?? 0);
        $eqCajasDisp = (float)($d['eq_cajas'] ?? 0);

        $sqlCua = "
          SELECT COALESCE(SUM(s.qty_uv),0) AS cajas, COALESCE(SUM(s.qty_uc),0) AS uc,
                 COALESCE(SUM( s.qty_uc / NULLIF(COALESCE(pu.unidades_por_uv, 0), 0) ), 0) AS eq_cajas
          FROM wh_stock s
          JOIN wh_pallet p ON p.id = s.pallet_id AND p.estado_id = {$cuaId}
          LEFT JOIN {$packUnitsAgg} pu ON pu.producto_id = s.producto_id
        ";
        $c = $pdo->query($sqlCua)->fetch(PDO::FETCH_ASSOC) ?: [];
        $cajasCuar   = (int)($c['cajas'] ?? 0);
        $ucCuar      = (int)($c['uc'] ?? 0);
        $eqCajasCuar = (float)($c['eq_cajas'] ?? 0);
      } else {
        $sqlDisp = "
          SELECT COALESCE(SUM(s.qty_uv),0) AS cajas, COALESCE(SUM(s.qty_uc),0) AS uc,
                 COALESCE(SUM( s.qty_uc / NULLIF(COALESCE(pu.unidades_por_uv, 0), 0) ), 0) AS eq_cajas
          FROM wh_stock s
          LEFT JOIN wh_posicion pos ON pos.id = s.posicion_id
          LEFT JOIN wh_ambiente amb ON amb.id = pos.ambiente_id
          LEFT JOIN {$packUnitsAgg} pu ON pu.producto_id = s.producto_id
          WHERE s.posicion_id IS NOT NULL AND (amb.code IS NULL OR amb.code <> 'CUARENTENA')
        ";
        $d = $pdo->query($sqlDisp)->fetch(PDO::FETCH_ASSOC) ?: [];
        $cajasDisp   = (int)($d['cajas'] ?? 0);
        $ucDisp      = (int)($d['uc'] ?? 0);
        $eqCajasDisp = (float)($d['eq_cajas'] ?? 0);

        $sqlCua = "
          SELECT COALESCE(SUM(s.qty_uv),0) AS cajas, COALESCE(SUM(s.qty_uc),0) AS uc,
                 COALESCE(SUM( s.qty_uc / NULLIF(COALESCE(pu.unidades_por_uv, 0), 0) ), 0) AS eq_cajas
          FROM wh_stock s
          JOIN wh_posicion pos ON pos.id = s.posicion_id
          JOIN wh_ambiente amb ON amb.id = pos.ambiente_id AND amb.code='CUARENTENA'
          LEFT JOIN {$packUnitsAgg} pu ON pu.producto_id = s.producto_id
        ";
        $c = $pdo->query($sqlCua)->fetch(PDO::FETCH_ASSOC) ?: [];
        $cajasCuar   = (int)($c['cajas'] ?? 0);
        $ucCuar      = (int)($c['uc'] ?? 0);
        $eqCajasCuar = (float)($c['eq_cajas'] ?? 0);
      }
    }
  } catch (Throwable $e) {
    // no-op
  }


  try {
    // ===== Pallets en Cuarentena =====
    $cuaId = (int)($pdo->query("SELECT id FROM wh_pallet_estado WHERE code='CUARENTENA' LIMIT 1")->fetchColumn() ?: 0);
    if ($cuaId) {
      // En CUARENTENA: definimos por estado_id (fuente de verdad)
      $stCua = $pdo->prepare("SELECT COUNT(*) FROM wh_pallet WHERE deleted_at IS NULL AND estado_id = ?");
      $stCua->execute([$cuaId]);
      $palletsCuarentena = (int)($stCua->fetchColumn() ?: 0);
    } else {
      $palletsCuarentena = 0;
    }
  } catch (Throwable $e) { $palletsCuarentena = 0; }


  try {
    // ===== Recepciones Pendientes =====
    // Consideramos estados previos a cierre/validación
    // Ajustá el set si tu flujo difiere
    $stRec = $pdo->query("
      SELECT COUNT(*) 
      FROM pl_packinglist 
      WHERE estado IN ('CREADO','IMPORTADO','EN_DESCARGA','DESCARGADO')
    ");
    $recepcionesPend = (int)($stRec->fetchColumn() ?: 0);
  } catch (Throwable $e) { 
    // Fallback alternativo: 0 si no existe la tabla
    $recepcionesPend = 0; 
  }

  try {
    // ===== Órdenes de Salida (Hoy) =====
    // Si existe columna orden_id, contamos distintas órdenes de salida hoy;
    // Si no, contamos movimientos OUT hoy.
    $ordenesSalidaHoy = 0;

    // Intento 1: contar por orden_id único
    $sqlOutByOrder = "SELECT COUNT(DISTINCT orden_id) FROM wh_move 
                      WHERE tipo='OUT' AND DATE(created_at)=? AND orden_id IS NOT NULL";
    $stOut1 = $pdo->prepare($sqlOutByOrder);
    $stOut1->execute([$hoy]);
    $ordenesSalidaHoy = (int)($stOut1->fetchColumn() ?: 0);

    // Si no hay columna orden_id o es 0, intentamos contar movimientos OUT
    if ($ordenesSalidaHoy === 0) {
      $stOut2 = $pdo->prepare("SELECT COUNT(*) FROM wh_move WHERE tipo='OUT' AND DATE(created_at)=?");
      $stOut2->execute([$hoy]);
      $ordenesSalidaHoy = (int)($stOut2->fetchColumn() ?: 0);
    }
  } catch (Throwable $e) {
    // Fallback si no existe wh_move
    $ordenesSalidaHoy = 0;
  }
}
?>
<div class="row g-3">
  <!-- Pallets en PICKING -->
  <div class="col-12 col-md-6 col-xl-3">
    <div class="card h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-center">
          <div>
            <p class="text-muted mb-1">Pallets en PICKING</p>
            <h4 class="mb-0"><?= number_format($palletsPicking, 0, ',', '.') ?></h4>
          </div>
          <i class="bi bi-basket fs-2"></i>
        </div>
        <small class="text-muted">Zona de picking</small>
      </div>
    </div>
  </div>

  <!-- Recepciones Pendientes -->
  <div class="col-12 col-md-6 col-xl-3">
    <div class="card h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-center">
          <div>
            <p class="text-muted mb-1">Recepciones Pendientes</p>
            <h4 class="mb-0"><?= number_format($recepcionesPend, 0, ',', '.') ?></h4>
          </div>
          <i class="bi bi-clipboard-check fs-2"></i>
        </div>
        <small class="text-warning">PL en proceso</small>
      </div>
    </div>
  </div>

  <!-- Órdenes de Salida (Hoy) -->
  <div class="col-12 col-md-6 col-xl-3">
    <div class="card h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-center">
          <div>
            <p class="text-muted mb-1">Órdenes de Salida (Hoy)</p>
            <h4 class="mb-0"><?= number_format($ordenesSalidaHoy, 0, ',', '.') ?></h4>
          </div>
          <i class="bi bi-truck fs-2"></i>
        </div>
        <small class="text-danger">Actividad del día</small>
      </div>
    </div>
  </div>

  <!-- Pallets en Cuarentena -->
  <div class="col-12 col-md-6 col-xl-3">
    <div class="card h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-center">
          <div>
            <p class="text-muted mb-1">Pallets en Cuarentena</p>
            <h4 class="mb-0"><?= number_format($palletsCuarentena, 0, ',', '.') ?></h4>
          </div>
          <i class="bi bi-shield-exclamation fs-2"></i>
        </div>
        <small class="text-warning">Pendientes de liberación</small>
      </div>
    </div>
  </div>

  

  <!-- Disponible: Cajas completas -->
  <div class="col-12 col-md-6 col-xl-3">
    <div class="card h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-center">
          <div>
            <p class="text-muted mb-1">Cajas disponibles</p>
            <h4 class="mb-0"><?= number_format($cajasDisp, 0, ',', '.') ?></h4>
          </div>
          <i class="bi bi-box fs-2"></i>
        </div>
        <small class="text-muted">UV fuera de Cuarentena</small>
      </div>
    </div>
  </div>

  <!-- Disponible: UC sueltas -->
  <div class="col-12 col-md-6 col-xl-3">
    <div class="card h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-center">
          <div>
            <p class="text-muted mb-1">UC sueltas disp.</p>
            <h4 class="mb-0"><?= number_format($ucDisp, 0, ',', '.') ?></h4>
          </div>
          <i class="bi bi-grid-3x3-gap fs-2"></i>
        </div>
        <small class="text-muted">≈ <?= number_format($eqCajasDisp, 2, ',', '.') ?> cajas</small>
      </div>
    </div>
  </div>

  <!-- Cuarentena: Cajas completas -->
  <div class="col-12 col-md-6 col-xl-3">
    <div class="card h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-center">
          <div>
            <p class="text-muted mb-1">Cajas en Cuarentena</p>
            <h4 class="mb-0"><?= number_format($cajasCuar, 0, ',', '.') ?></h4>
          </div>
          <i class="bi bi-shield fs-2"></i>
        </div>
        <small class="text-muted">UV en Cuarentena</small>
      </div>
    </div>
  </div>

  <!-- Cuarentena: UC sueltas -->
  <div class="col-12 col-md-6 col-xl-3">
    <div class="card h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-center">
          <div>
            <p class="text-muted mb-1">UC sueltas en Cuarentena</p>
            <h4 class="mb-0"><?= number_format($ucCuar, 0, ',', '.') ?></h4>
          </div>
          <i class="bi bi-shield fs-2"></i>
        </div>
        <small class="text-muted">≈ <?= number_format($eqCajasCuar, 2, ',', '.') ?> cajas</small>
      </div>
    </div>
  </div>
</div>
<div class="row g-3 mt-0 mb-2">
  <!-- Peso disponible (kg) -->
  <div class="col-12 col-md-6 col-xl-3">
    <div class="card h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-center">
          <div>
            <p class="text-muted mb-1">Peso disponible (kg)</p>
            <h4 class="mb-0"><?= number_format($pesoDispKg, 0, ',', '.') ?></h4>
          </div>
          <i class="bi bi-speedometer2 fs-2"></i>
        </div>
        <small class="text-muted">UC x peso estándar (excluye Cuarentena/Embarcado)</small>
      </div>
    </div>
  </div>

  <!-- Valor disponible del stock (Gs) -->
  <div class="col-12 col-md-6 col-xl-3">
    <div class="card h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-center">
          <div>
            <p class="text-muted mb-1">Valor disponible (Gs)</p>
            <h4 class="mb-0"><?= number_format($valorDispGs, 0, ',', '.') ?></h4>
          </div>
          <i class="bi bi-currency-exchange fs-2"></i>
        </div>
        <small class="text-muted">UC x precio efectivo (excluye Cuarentena/Embarcado)</small>
      </div>
    </div>
  </div>

  <!-- Peso en Cuarentena (kg) -->
  <div class="col-12 col-md-6 col-xl-3">
    <div class="card h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-center">
          <div>
            <p class="text-muted mb-1">Peso en Cuarentena (kg)</p>
            <h4 class="mb-0"><?= number_format($pesoCuarKg, 0, ',', '.') ?></h4>
          </div>
          <i class="bi bi-shield fs-2"></i>
        </div>
        <small class="text-muted">UC x peso estándar</small>
      </div>
    </div>
  </div>

  <!-- Valor en Cuarentena (Gs) -->
  <div class="col-12 col-md-6 col-xl-3">
    <div class="card h-100">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-center">
          <div>
            <p class="text-muted mb-1">Valor en Cuarentena (Gs)</p>
            <h4 class="mb-0"><?= number_format($valorCuarGs, 0, ',', '.') ?></h4>
          </div>
          <i class="bi bi-shield fs-2"></i>
        </div>
        <small class="text-muted">UC x precio efectivo</small>
      </div>
    </div>
  </div>
</div>
