<?php
declare(strict_types=1);
/**
 * Tabla resumida de actividad reciente.
 */

if (!function_exists('e')) {
  function e(string $s): string
  {
    return htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
  }
}

if (!function_exists('project_path')) {
  function project_path(string $rel): string
  {
    return rtrim(dirname(__DIR__, 4), DIRECTORY_SEPARATOR) . DIRECTORY_SEPARATOR . ltrim($rel, DIRECTORY_SEPARATOR);
  }
}

require_once project_path('config/db.php');

$pdo = null;
try {
  $pdo = getPDO();
} catch (Throwable $e) {
  error_log('[dashboard.ultimos_movimientos] DB init error: ' . $e->getMessage());
}

$movimientos = [];

if ($pdo instanceof PDO) {
  foreach ([
    'SELECT * FROM wh_move ORDER BY created_at DESC, id DESC LIMIT 10',
    'SELECT * FROM wh_move ORDER BY id DESC LIMIT 10',
  ] as $sqlCandidate) {
    try {
      $st = $pdo->query($sqlCandidate);
      if ($st) {
        $movimientos = $st->fetchAll(PDO::FETCH_ASSOC) ?: [];
      }
    } catch (Throwable $e) {
      $movimientos = [];
      continue;
    }
    if (!empty($movimientos)) {
      break;
    }
  }

  if (!empty($movimientos)) {
    $palletIds = [];
    $posIds = [];
    $userIds = [];

    foreach ($movimientos as $row) {
      $pid = (int) ($row['pallet_id'] ?? 0);
      if ($pid > 0) {
        $palletIds[] = $pid;
      }
      $from = (int) ($row['from_pos_id'] ?? 0);
      if ($from > 0) {
        $posIds[] = $from;
      }
      $to = (int) ($row['to_pos_id'] ?? 0);
      if ($to > 0) {
        $posIds[] = $to;
      }
      foreach (['created_by', 'usuario_id', 'user_id'] as $candidate) {
        if (isset($row[$candidate]) && (int) $row[$candidate] > 0) {
          $userIds[] = (int) $row[$candidate];
          break;
        }
      }
    }

    $palletIds = array_values(array_unique($palletIds));
    $posIds = array_values(array_unique($posIds));
    $userIds = array_values(array_unique($userIds));

    $palletMap = [];
    if ($palletIds) {
      try {
        $placeholders = implode(',', array_fill(0, count($palletIds), '?'));
        $st = $pdo->prepare('SELECT * FROM wh_pallet WHERE id IN (' . $placeholders . ')');
        $st->execute($palletIds);
        foreach ($st as $row) {
          $id = (int) ($row['id'] ?? 0);
          if ($id <= 0) {
            continue;
          }
          $label = null;
          foreach (['codigo', 'code_full', 'code', 'codigo_full'] as $col) {
            if (array_key_exists($col, $row) && $row[$col]) {
              $label = (string) $row[$col];
              break;
            }
          }
          if ($label === null) {
            $label = '#' . $id;
          }
          $palletMap[$id] = $label;
        }
      } catch (Throwable $e) {
        $palletMap = [];
      }
    }

    $posMap = [];
    if ($posIds) {
      try {
        $placeholders = implode(',', array_fill(0, count($posIds), '?'));
        $st = $pdo->prepare('SELECT * FROM wh_posicion WHERE id IN (' . $placeholders . ')');
        $st->execute($posIds);
        foreach ($st as $row) {
          $id = (int) ($row['id'] ?? 0);
          if ($id <= 0) {
            continue;
          }
          $label = null;
          foreach (['code_full', 'code', 'pos_code_full', 'pos_code', 'titulo', 'title'] as $col) {
            if (array_key_exists($col, $row) && $row[$col]) {
              $label = (string) $row[$col];
              break;
            }
          }
          if ($label === null) {
            $label = '#' . $id;
          }
          $posMap[$id] = $label;
        }
      } catch (Throwable $e) {
        $posMap = [];
      }
    }

    $userMap = [];
    if ($userIds) {
      try {
        $placeholders = implode(',', array_fill(0, count($userIds), '?'));
        $st = $pdo->prepare('SELECT id, username, full_name FROM sys_users WHERE id IN (' . $placeholders . ')');
        $st->execute($userIds);
        foreach ($st as $row) {
          $id = (int) ($row['id'] ?? 0);
          if ($id <= 0) {
            continue;
          }
          $name = (string) ($row['full_name'] ?? '');
          $login = (string) ($row['username'] ?? '');
          if ($name === '' && $login === '') {
            $name = '#' . $id;
            $login = $name;
          }
          $userMap[$id] = ['name' => $name !== '' ? $name : $login, 'login' => $login !== '' ? $login : $name];
        }
      } catch (Throwable $e) {
        $userMap = [];
      }
    }

    foreach ($movimientos as &$row) {
      $row['created_by_id'] = null;
      foreach (['created_by', 'usuario_id', 'user_id'] as $candidate) {
        if (isset($row[$candidate]) && (int) $row[$candidate] > 0) {
          $row['created_by_id'] = (int) $row[$candidate];
          break;
        }
      }

      $pid = (int) ($row['pallet_id'] ?? 0);
      if ($pid > 0 && isset($palletMap[$pid])) {
        $row['pallet_codigo'] = $palletMap[$pid];
      }
      $fromId = (int) ($row['from_pos_id'] ?? 0);
      if ($fromId > 0 && isset($posMap[$fromId])) {
        $row['from_code'] = $posMap[$fromId];
      }
      $toId = (int) ($row['to_pos_id'] ?? 0);
      if ($toId > 0 && isset($posMap[$toId])) {
        $row['to_code'] = $posMap[$toId];
      }
      $uid = (int) ($row['created_by_id'] ?? 0);
      if ($uid > 0 && isset($userMap[$uid])) {
        $row['usuario_nombre'] = $userMap[$uid]['name'];
        $row['usuario_login'] = $userMap[$uid]['login'];
      }
    }
    unset($row);
  }
}

$tipoLabels = [
  'IN'         => 'Entrada',
  'OUT'        => 'Salida',
  'MOVE'       => 'Reubicación',
  'MOVE_IN'    => 'Movimiento (+)',
  'MOVE_OUT'   => 'Movimiento (-)',
  'ADJUST'     => 'Ajuste',
  'COUNT'      => 'Conteo',
  'TRANSFER'   => 'Transferencia',
];

$fmtDateTime = static function (?string $value): string {
  if (!$value) {
    return '—';
  }
  $ts = strtotime($value);
  if ($ts === false) {
    return $value;
  }
  return date('Y-m-d H:i', $ts);
};

$renderTipo = static function (?string $tipo) use ($tipoLabels): string {
  if (!$tipo) {
    return '—';
  }
  $key = strtoupper($tipo);
  if (isset($tipoLabels[$key])) {
    return $tipoLabels[$key];
  }
  return ucfirst(strtolower($tipo));
};

$formatPallet = static function (array $row): string {
  if (!empty($row['pallet_codigo'])) {
    return (string) $row['pallet_codigo'];
  }
  if (!empty($row['pallet_id'])) {
    return '#' . $row['pallet_id'];
  }
  return '—';
};

$formatPos = static function (?string $code, $id): string {
  if ($code) {
    return (string) $code;
  }
  if ($id !== null && $id !== '') {
    return '#' . $id;
  }
  return '—';
};

$formatUsuario = static function (array $row): string {
  if (!empty($row['usuario_nombre'])) {
    return (string) $row['usuario_nombre'];
  }
  if (!empty($row['usuario_login'])) {
    return (string) $row['usuario_login'];
  }
  if (!empty($row['created_by_id'])) {
    return '#' . $row['created_by_id'];
  }
  return '—';
};
?>
<div class="row g-3 mt-1">
  <div class="col-12">
    <div class="card">
      <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0">Últimos movimientos</h5>
        <a class="btn btn-sm btn-outline-secondary" href="<?= e(url('/operaciones/movimientos')) ?>">
          Ver todo
        </a>
      </div>
      <div class="card-body p-0">
        <div class="table-responsive">
          <table class="table table-hover table-sm mb-0 align-middle">
            <thead class="table-light">
              <tr>
                <th class="text-nowrap">Fecha/Hora</th>
                <th class="text-nowrap">Tipo</th>
                <th class="text-nowrap">Pallet</th>
                <th class="text-nowrap">Desde</th>
                <th class="text-nowrap">Hacia</th>
                <th class="text-nowrap">Usuario</th>
              </tr>
            </thead>
            <tbody>
              <?php if (!empty($movimientos)): ?>
                <?php foreach ($movimientos as $row): ?>
                  <?php
                  $fecha = $fmtDateTime($row['created_at'] ?? null);
                  $tipo = $renderTipo($row['tipo'] ?? null);
                  $motivo = trim((string) ($row['motivo'] ?? ''));
                  $pallet = $formatPallet($row);
                  $desde = $formatPos($row['from_code'] ?? null, $row['from_pos_id'] ?? null);
                  $hasta = $formatPos($row['to_code'] ?? null, $row['to_pos_id'] ?? null);
                  $usuario = $formatUsuario($row);
                  $referencia = trim((string) ($row['referencia'] ?? ''));
                  ?>
                  <tr>
                    <td class="text-nowrap" title="<?= $referencia !== '' ? e($referencia) : '' ?>"><?= e($fecha) ?></td>
                    <td class="text-nowrap">
                      <span class="fw-semibold"><?= e($tipo) ?></span>
                      <?php if ($motivo !== ''): ?>
                        <span class="badge bg-light text-dark ms-1"><?= e($motivo) ?></span>
                      <?php endif; ?>
                    </td>
                    <td class="text-nowrap"><?= e($pallet) ?></td>
                    <td class="text-nowrap"><?= e($desde) ?></td>
                    <td class="text-nowrap"><?= e($hasta) ?></td>
                    <td class="text-nowrap"><?= e($usuario) ?></td>
                  </tr>
                <?php endforeach; ?>
              <?php else: ?>
                <tr>
                  <td colspan="6" class="text-center py-4 text-muted">Sin movimientos registrados.</td>
                </tr>
              <?php endif; ?>
            </tbody>
          </table>
        </div>
      </div>
      <div class="card-footer text-muted small">
        Fuente: wh_move (últimos 10 registros ordenados por fecha más reciente)
      </div>
    </div>
  </div>
</div>
