<?php
declare(strict_types=1);

/**
 * SOL - Sistema de Operaciones Logísticas
 * Vista: views/control/mapeo/index.php
 * Módulo: Control → Mapeo por ubicación (layout de depósito)
 *
 * - Renderiza el layout del depósito (rack_layout.php).
 * - Carga CSS específico de warehouse y los JS de layout-core + renderers.
 * - Bootstrap de página: public/page-scripts/pages/control/mapeo.js
 *
 * Requisitos previos:
 * - Helpers: app/Support/ViewHelpers.php, app/Support/Assets.php
 * - Parcial de layout: views/layout/_partials/warehouse/rack_layout.php
 */

$BASE = dirname(__DIR__, 3);

require_once $BASE . '/app/Support/ViewHelpers.php';
require_once $BASE . '/app/Support/Assets.php';

// -------------------------
// Meta
// -------------------------
sol_section('page_title', 'Control · Mapeo por ubicación');

// -------------------------
// Lee params (opcionales) para inicializar la vista
// -------------------------
$fechaParam      = isset($_GET['fecha']) ? (string)$_GET['fecha'] : '';
$depositoIdParam = isset($_GET['deposito_id']) ? (int)$_GET['deposito_id'] : 0;
$rackParam       = isset($_GET['rack']) ? (string)$_GET['rack'] : '';
$viewParam       = isset($_GET['view']) ? (string)$_GET['view'] : 'microtiles';

// Normaliza rack aceptando "5" o "R05"
$rackNum = 0;
if ($rackParam !== '') {
  if (preg_match('/^\s*[Rr]\s*0*([0-9]+)\s*$/', $rackParam, $m)) {
    $rackNum = (int)$m[1];
  } else {
    $rackNum = (int)$rackParam;
  }
}

// Estructuras mínimas para el parcial (pueden quedar vacías)
$deposito = ($depositoIdParam > 0)
  ? ['id' => $depositoIdParam, 'code' => 'DEP' . $depositoIdParam, 'name' => null]
  : ['id' => null, 'code' => '', 'name' => null];

$rack = ($rackNum > 0)
  ? ['id' => $rackNum, 'code' => 'R' . str_pad((string)$rackNum, 2, '0', STR_PAD_LEFT), 'name' => null]
  : ['id' => null, 'code' => '', 'name' => null];

// Sin posiciones server-side: las cargará el JS por API cuando el usuario elija depósito/rack
$positions = [];

// -------------------------
// Assets específicos
// -------------------------
// 1) CSS: layout de depósito (warehouse.scss → warehouse.css)
// 2) JS: primero core, luego renderers, por último bootstrap de la página
sol_assets_push(
  [
    url('assets/css/warehouse.css'),
  ],
  [
    url('page-scripts/warehouse/layout-core.js'),
    url('page-scripts/warehouse/render-microtiles.js'),
    url('page-scripts/warehouse/render-shelf-strips.js'),
    url('page-scripts/warehouse/render-column-cards.js'),
    url('page-scripts/pages/control/mapeo.js'),
  ]
);

// -------------------------
// Contenido principal
// -------------------------
ob_start();
$WAREHOUSE_LAYOUT = [
  'form_id'      => 'frmConteo',   // IDs usados por los JS
  'show_filters' => true,          // muestra select Depósito/Rack/Vista
  'show_date'    => false,         // ocultar fecha en MAPEO
  'show_legend'  => true,          // mostrar leyenda
  // 'default_view' => 'microtiles', // opcional
];
?>
<div class="container-fluid">

  <!-- Cabecera de filtros / controles (ahora vive en la vista, no en el parcial) -->
  <div class="card mb-3">
    <div class="card-body">
      <form id="frmMapeo" class="row g-3">

        <!-- Depósito (se cargará por API meta=depositos) -->
        <div class="col-12 col-md-3">
          <label for="flt-deposito" class="form-label">Depósito</label>
          <select id="flt-deposito" name="deposito_id" class="form-select" data-source="api/control/conteo_layout.php?meta=depositos">
            <?php if ($deposito['id']): ?>
              <option value="<?= (int)$deposito['id'] ?>" selected>DEP<?= (int)$deposito['id'] ?></option>
            <?php else: ?>
              <option value="">Seleccione...</option>
            <?php endif; ?>
          </select>
        </div>

        <!-- Rack (depende de depósito: api/control/conteo_layout.php?meta=racks&deposito_id=###) -->
        <div class="col-12 col-md-3">
          <label for="flt-rack" class="form-label">Rack</label>
          <select id="flt-rack" name="rack" class="form-select" data-source-base="api/control/conteo_layout.php?meta=racks&deposito_id=">
            <?php if ($rack['id']): ?>
              <option value="<?= htmlspecialchars($rack['code']) ?>" selected><?= htmlspecialchars($rack['code']) ?></option>
            <?php else: ?>
              <option value="">Seleccione...</option>
            <?php endif; ?>
          </select>
        </div>

        <!-- Modo de vista -->
        <div class="col-12 col-md-2">
          <label for="flt-view" class="form-label">Vista</label>
          <select id="flt-view" name="view" class="form-select" data-role="layout-mode">
            <?php
            $view = strtolower($viewParam) ?: 'microtiles';
            $opts = [
              'microtiles'    => 'Microtiles',
              'shelf-strips'  => 'Shelf strips',   // ← antes 'shelf_strips'
              'column-cards'  => 'Column cards',   // ← antes 'column_cards'
              'cubes'         => 'Cubes',
            ];
            foreach ($opts as $k => $label) {
              $sel = ($k === $view) ? 'selected' : '';
              echo "<option value=\"{$k}\" {$sel}>{$label}</option>";
            }
            ?>
          </select>

        </div>

        <!-- Mostrar leyenda -->
        <!-- <div class="col-12 col-md-1">
          <div class="form-check mt-4">
            <input class="form-check-input" type="checkbox" value="0" id="chk-legend" checked>
            <label class="form-check-label" for="chk-legend">Leyenda</label>
          </div>
        </div> -->

        <!-- Botón aplicar -->
        <div class="col-12 col-md-12 d-flex gap-2 justify-content-end">
          <button type="submit" class="btn btn-primary">Aplicar</button>
        </div>
      </form>
    </div>
  </div>

  <?php
    // El parcial dibuja el contenedor del layout (filtros + área de render)
    include project_path('views/layout/_partials/warehouse/rack_layout.php');
  ?>
</div>
<?php
$content = ob_get_clean();

// -------------------------
// Render del layout
// -------------------------
include project_path('views/layout/layout_app.php');
