<?php
declare(strict_types=1);

/**
 * SOL - Sistema de Operaciones Logísticas
 * Parcial: views/layout/_partials/sidebar.php
 *
 * Sidebar dinámico desde config/menu.php.
 * - Dropdown SOLO si hay hijos.
 * - Marca activo y abre el grupo según la ruta actual.
 * - Fuerza apertura con inline style + d-block (compat Viho sidebar-menu.js).
 */

if (!defined('BASE_PATH')) {
  define('BASE_PATH', dirname(__DIR__, 3));
}

// -----------------------
// Cargar helpers del menú
// -----------------------
if (!function_exists('menu_load')) {
  $maybeRouter = BASE_PATH . '/app/Support/MenuRouter.php';
  if (is_file($maybeRouter)) require_once $maybeRouter;
}

$menu = [];
try {
  if (function_exists('menu_load')) {
    $menu = menu_load();
  } else {
    $file = BASE_PATH . '/config/menu.php';
    if (!is_file($file)) throw new RuntimeException("No se encontró config/menu.php en: {$file}");
    $menu = require $file;
  }
} catch (Throwable $e) {
  $menu = [];
}

// -----------------------
// Helpers de path y activo
// -----------------------
if (!function_exists('sol_current_path')) {
  function sol_current_path(): string {
    $p = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH) ?? '/';
    $p = rtrim($p, '/');
    return $p === '' ? '/' : $p;
  }
}

if (!function_exists('sol_item_visible')) {
  function sol_item_visible(array $it): bool {
    return ($it['visible'] ?? true) === true;
  }
}

if (!function_exists('sol_has_children')) {
  function sol_has_children(array $it): bool {
    return isset($it['children']) && is_array($it['children']) && count($it['children']) > 0;
  }
}

if (!function_exists('sol_is_section')) {
  function sol_is_section(array $it): bool {
    return ($it['section'] ?? false) === true;
  }
}

if (!function_exists('sol_is_active')) {
  /**
   * Activo si coincide exactamente o si $current empieza por $url + '/'.
   */
  function sol_is_active(?string $url, string $current): bool {
    if (!$url) return false;
    $u = rtrim($url, '/'); if ($u === '') $u = '/';
    $c = rtrim($current, '/'); if ($c === '') $c = '/';
    if ($u === $c) return true;
    if ($u !== '/' && str_starts_with($c, $u . '/')) return true;
    return false;
  }
}

if (!function_exists('sol_group_active')) {
  /**
   * Un grupo está activo si algún hijo visible está activo (recursivo).
   */
  function sol_group_active(array $children, string $current): bool {
    foreach ($children as $c) {
      if (!sol_item_visible($c)) continue;
      $u = $c['url'] ?? null;
      if (is_string($u) && sol_is_active($u, $current)) return true;
      if (sol_has_children($c) && sol_group_active($c['children'], $current)) return true;
    }
    return false;
  }
}

$foto = htmlspecialchars(url('/assets/img/personal/'.$_SESSION['usuario_id'].'.jpg'));

$currentPath = sol_current_path();

?>
<header class="main-nav">
  <!-- Bloque usuario (placeholder) -->
  <div class="sidebar-user text-center">
    <img class="img-90 rounded-circle" src="<?= $foto ?>" alt="user">
    <div class="badge-bottom mt-3"><span class="badge badge-primary"><?= htmlspecialchars($_SESSION['usuario_rol'] ?? 'Usuario') ?></span></div>
    <a href="javascript:void(0)"><h6 class="mt-3 f-14 f-w-600 mb-3"><?= htmlspecialchars($_SESSION['usuario_nombre'] ?? 'Usuario') ?></h6></a>
    <!-- <p class="mb-0 font-roboto"><?= htmlspecialchars($_SESSION['usuario_rol'] ?? 'Operador') ?></p> -->
  </div>

  <nav>
    <div class="main-navbar">
      <div class="left-arrow" id="left-arrow"><i data-feather="arrow-left"></i></div>
      <div id="mainnav">
        <ul class="nav-menu custom-scrollbar">
          <li class="back-btn">
            <div class="mobile-back text-end"><span>Volver</span><i class="fa fa-angle-right ps-2" aria-hidden="true"></i></div>
          </li>

          <?php foreach ($menu as $item): ?>
            <?php if (!sol_item_visible($item)) continue; ?>

            <?php if (sol_is_section($item)): ?>
              <!-- Título de sección -->
              <li class="sidebar-main-title">
                <div><h6><?= htmlspecialchars($item['label'] ?? '') ?></h6></div>
              </li>

            <?php else: ?>
              <?php
                $icon     = (string)($item['icon'] ?? 'circle');
                $label    = (string)($item['label'] ?? '');
                $urlItem  = $item['url'] ?? null;
                $children = $item['children'] ?? [];
                $isGroup  = sol_has_children($item); // SOLO dropdown si tiene hijos
                $isOpen   = $isGroup ? sol_group_active($children, $currentPath)
                                     : sol_is_active(is_string($urlItem) ? $urlItem : null, $currentPath);
              ?>

              <?php if ($isGroup): ?>
                <!-- Grupo con hijos (dropdown) -->
                <?php
                  $liClass   = 'dropdown' . ($isOpen ? ' active' : '');
                  $aClass    = 'nav-link menu-title' . ($isOpen ? ' active' : '');
                  $submenuSt = $isOpen ? 'display:block' : 'display:none';
                  $submenuCl = 'nav-submenu menu-content' . ($isOpen ? ' d-block' : '');
                ?>
                <li class="<?= $liClass ?>">
                  <a class="nav-link menu-title active" href="javascript:void(0)" aria-expanded="<?= $isOpen ? 'true' : 'false' ?>">
                    <i data-feather="<?= htmlspecialchars($icon) ?>"></i>
                    <span><?= htmlspecialchars($label) ?></span>
                    <div class="according-menu<?= $isOpen ? ' active' : '' ?>"><i class="fa fa-angle-right"></i></div>
                  </a>
                  <ul class="<?= $submenuCl ?>" style="<?= $submenuSt ?>;">
                    <?php foreach ($children as $ch): ?>
                      <?php
                        if (!sol_item_visible($ch)) continue;
                        $chUrl   = $ch['url']   ?? null;
                        $chLabel = $ch['label'] ?? '';
                        if (!is_string($chUrl) || $chUrl === '') continue;
                        $isChildActive = sol_is_active($chUrl, $currentPath);
                      ?>
                      <li class="<?= $isChildActive ? 'active' : '' ?>">
                        <a href="<?= htmlspecialchars(url($chUrl)) ?>" class="<?= $isChildActive ? 'active' : '' ?>" <?= $isChildActive ? 'aria-current="page"' : '' ?>>
                          <?= htmlspecialchars($chLabel) ?>
                        </a>
                      </li>
                    <?php endforeach; ?>
                  </ul>
                </li>

              <?php else: ?>
                <!-- Ítem simple (link directo) -->
                <?php $isActive = sol_is_active(is_string($urlItem) ? $urlItem : null, $currentPath); ?>
                <li class="<?= $isActive ? 'active' : '' ?>">
                  <a class="nav-link menu-title<?= $isActive ? ' active' : '' ?>"
                     href="<?= htmlspecialchars(url((string)$urlItem)) ?>"
                     <?= $isActive ? 'aria-current="page"' : '' ?>>
                    <i data-feather="<?= htmlspecialchars($icon) ?>"></i>
                    <span><?= htmlspecialchars($label) ?></span>
                  </a>
                </li>
              <?php endif; ?>
            <?php endif; ?>

          <?php endforeach; ?>
        </ul>
      </div>
      <div class="right-arrow" id="right-arrow"><i data-feather="arrow-right"></i></div>
    </div>
  </nav>
</header>
