<?php

declare(strict_types=1);

/**
 * SOL - Sistema de Operaciones Logísticas
 * Parcial: views/layout/_partials/warehouse/rack_layout.php
 *
 * Propósito:
 *   Dispatcher reusable para visualizar un rack en distintos modos:
 *   - microtiles
 *   - shelf_strips
 *   - column_cards
 *   - cubes
 *
 * Entradas esperadas:
 *   - $deposito: array{id:int, code:string, name?:string}
 *   - $rack:     array{id:int, code:string, name?:string, lado_code?:string, orientacion_code?:string}
 *   - $positions: array<array{
 *       id?:int,
 *       col_code:string,          // Ej: "C01"
 *       niv_code:string,          // Ej: "N01"
 *       depth_index:int,          // Ej: 1..n (fondo)
 *       code_full:string,         // Ej: "DEP-RK-C01-N01-F01"
 *       status:'ACTIVE'|'BLOCKED'|'INACTIVE',
 *       is_pick_face:int,         // 0/1
 *       blocked_reason?:?string,
 *       fondo_id?:int             // recomendado (id posición/fondo)
 *   }>
 *   - $options (opcional):
 *       mode: string ('microtiles'|'shelf_strips'|'column_cards'|'cubes')
 *       show_toolbar: bool (por defecto true)
 *       title: string (título a mostrar arriba)
 *
 * Notas:
 *   - Este archivo NO incluye CSS/JS. Los estilos y scripts del layout
 *     se integrarán después (tus SCSS/JS). Por ahora solo emite HTML/atributos.
 *   - Los sub-parciales deben ubicarse en el mismo directorio con nombres:
 *       _rack_layout.microtiles.php
 *       _rack_layout.shelf_strips.php
 *       _rack_layout.column_cards.php
 *       _rack_layout.cubes.php
 */

if (!function_exists('h')) {
  function h($s): string
  {
    return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8');
  }
}

// -------------------------
// Validación mínima de entradas
// -------------------------
$deposito  = is_array($deposito ?? null) ? $deposito : [];
$rack      = is_array($rack ?? null) ? $rack : [];
$positions = is_array($positions ?? null) ? $positions : [];
$options   = is_array($options ?? null) ? $options : [];

// Identificadores
$depCode  = (string)($deposito['code'] ?? '');
$rackCode = (string)($rack['code']     ?? '');

// -------------------------
// Normalización de "mode"
// -------------------------
$modeReq = strtolower((string)($options['mode'] ?? ($_GET['view'] ?? '')));
$aliases = [
  'micro'        => 'microtiles',
  'microtiles'   => 'microtiles',
  'shelf'        => 'shelf_strips',
  'shelf-strips' => 'shelf_strips',
  'shelf_strips' => 'shelf_strips',
  'column'       => 'column_cards',
  'cards'        => 'column_cards',
  'cubes'        => 'cubes'
];
$mode = $aliases[$modeReq] ?? 'microtiles';

// -------------------------
// Estructuras derivadas para render
// layout[dep][rack][col][niv] = [depth_index, depth_index, ...]
// posMap[dep][rack][col][niv][depth_index] = fondo_id|position_id
// -------------------------
$layout = [];
$posMap = [];
$colsSet = [];
$nivSet  = [];

foreach ($positions as $p) {
  $col   = (string)($p['col_code'] ?? '');
  $niv   = (string)($p['niv_code'] ?? '');
  $f     = (int)($p['depth_index'] ?? 0);
  $fid   = $p['fondo_id'] ?? ($p['id'] ?? null);

  if ($col === '' || $niv === '' || $f <= 0) continue;

  $layout[$depCode][$rackCode][$col][$niv]   ??= [];
  $posMap[$depCode][$rackCode][$col][$niv]   ??= [];
  $posMap[$depCode][$rackCode][$col][$niv][$f] = $fid;

  // Para mantener el orden y evitar duplicados
  if (!isset($colsSet[$col])) $colsSet[$col] = true;
  if (!isset($nivSet[$niv]))  $nivSet[$niv]  = true;

  // Lista de fondos por nivel/columna (sin duplicados)
  if (!in_array($f, $layout[$depCode][$rackCode][$col][$niv], true)) {
    $layout[$depCode][$rackCode][$col][$niv][] = $f;
  }
}

// Orden natural: C01..C10, N1..N10
$cols = array_keys($colsSet);
$niv  = array_keys($nivSet);
usort($cols, 'strnatcmp');
usort($niv,  'strnatcmp');

// -------------------------
// Toolbar opcional
// -------------------------
$showToolbar = array_key_exists('show_toolbar', $options) ? (bool)$options['show_toolbar'] : true;
$title       = (string)($options['title'] ?? ("Rack " . ($rack['name'] ?? $rackCode)));

?>
<div class="card">
  <div class="card-body">
    <div class="warehouse-layout" data-dep="<?= h($depCode) ?>" data-rack="<?= h($rackCode) ?>" data-mode="<?= h($mode) ?>">
      <?php if ($showToolbar): ?>
        <div class="d-flex align-items-center justify-content-between mb-3 flex-wrap gap-2">
          <div class="d-flex align-items-center gap-2">
            <div class="fw-semibold"><?= h($title) ?></div>
            <span class="badge text-bg-light"><?= h($depCode) ?></span>
            <span class="badge text-bg-secondary"><?= h($rackCode) ?></span>
          </div>

          <div class="d-flex align-items-center gap-2">
            <!-- Select de modo (no enlazamos JS aún) -->
            <select class="form-select form-select-sm" name="layout_view" data-role="layout-mode">
              <option value="microtiles" <?= $mode === 'microtiles' ? 'selected' : '' ?>>Microtiles</option>
              <option value="shelf_strips" <?= $mode === 'shelf_strips' ? 'selected' : '' ?>>Shelf strips</option>
              <option value="column_cards" <?= $mode === 'column_cards' ? 'selected' : '' ?>>Column cards</option>
              <option value="cubes" <?= $mode === 'cubes' ? 'selected' : '' ?>>Cubes</option>
            </select>
          </div>
        </div>
      <?php endif; ?>

      <div class="warehouse-layout__canvas">
        <?php
        // Disponibilizamos variables para el sub-parcial
        $__vars = [
          'layout'    => $layout,
          'posMap'    => $posMap,
          'positions' => $positions,
          'depCode'   => $depCode,
          'rackCode'  => $rackCode,
          'cols'      => $cols,
          'niv'       => $niv
        ];

        foreach ($__vars as $__k => $__v) {
          $$__k = $__v;
        }

        $baseDir = __DIR__;
        $partials = [
          'microtiles'   => $baseDir . '/_rack_layout.microtiles.php',
          'shelf_strips' => $baseDir . '/_rack_layout.shelf_strips.php',
          'column_cards' => $baseDir . '/_rack_layout.column_cards.php',
          'cubes'        => $baseDir . '/_rack_layout.cubes.php',
        ];

        if (is_file($partials[$mode] ?? '')) {
          include $partials[$mode];
        } else {
          // Fallback mínimo si no existen aún los sub-parciales
        ?>
          <div class="alert alert-warning">
            <div class="fw-semibold mb-1">Vista no disponible</div>
            <div>El sub-parcial para el modo <code><?= h($mode) ?></code> no existe. Crea el archivo: <code><?= h(basename($partials[$mode] ?? '')) ?></code>.</div>
          </div>
        <?php
        }
        ?>
      </div>
    </div>

  </div>
</div>