<?php
/**
 * SOL - Sistema de Operaciones Logísticas
 * Layout principal: views/layout/layout_app.php
 *
 * Usa partials en views/layout/_partials:
 *   - header.php   → <head> y CSS
 *   - topbar.php   → barra superior
 *   - sidebar.php  → menú lateral
 *   - footer.php   → scripts JS
 *
 * Variables que pueden definirse desde cada vista:
 *   - page_title (string)
 *   - page_toolbar (HTML)
 *   - page_css[] (array)
 *   - page_js[] (array)
 *   - page_scripts_end (HTML)
 *
 * Contenido:
 *   - $content (string) o $renderBody (callable)
 */

declare(strict_types=1);

// Helpers mínimos (por si la vista los necesita)
if (!function_exists('env')) {
  function env(string $key, $default = null) {
    return $_ENV[$key] ?? getenv($key) ?: $default;
  }
}
if (!function_exists('url')) {
  function url(string $path = '/'): string {
    $base = rtrim((string)(env('BASE_URL', '')), '/');
    $path = '/' . ltrim($path, '/');
    return $base . $path;
  }
}
if (!function_exists('project_path')) {
  function project_path(string $rel): string {
    return rtrim(dirname(__DIR__, 2), DIRECTORY_SEPARATOR)
         . DIRECTORY_SEPARATOR
         . str_replace(['/', '\\'], DIRECTORY_SEPARATOR, ltrim($rel, '/'));
  }
}
if (!function_exists('render_section')) {
  function render_section(string $name, $default = ''): void {
    $sections = $GLOBALS['__sol_sections'] ?? [];
    if (!array_key_exists($name, $sections)) {
      echo $default;
      return;
    }
    $val = $sections[$name];
    if (is_array($val)) {
      echo implode("\n", array_map(static fn($v) => (string)$v, $val));
    } else {
      echo (string)$val;
    }
  }
}

// ===========================================================
// GATE DE SESIÓN (CENTRALIZADO)
// - Permite /login sin sesión; exige sesión para el resto.
// - Colocado ANTES de cualquier salida para poder usar header().
// ===========================================================
if (session_status() !== PHP_SESSION_ACTIVE) {
  @session_start();
}
$__reqPath  = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH) ?? '/';
$__isLogin  = (bool)preg_match('#^/login\b#', $__reqPath);

if (empty($_SESSION['usuario_id']) && !$__isLogin) {
  if (!headers_sent()) {
    header('Location: ' . url('/login') . '?e=' . urlencode('Debes iniciar sesión.'));
    exit;
  }
  echo '<script>location.href=' . json_encode(url('/login') . '?e=Debes%20iniciar%20sesion') . ';</script>';
  exit;
}

// Datos de entorno
$appName        = (string) env('PROYECTO_TITULO', env('APP_NAME', 'SOL'));
$logoLight      = (string) env('APP_LOGO', url('/assets/img/logo-light.png'));
$logoDark       = (string) env('APP_LOGO_DARK',  url('/assets/img/logo-dark.png'));
$pageTitle      = $GLOBALS['__sol_sections']['page_title'] ?? $appName;

// Toolbar por defecto con bookmarks
$urlControl = url('/control');
$urlReportes = url('/reportes');
$urlParametros = url('/parametros');
$defaultToolbar = <<<HTML
<!-- Bookmark Start-->
<div class="bookmark">
  <ul>
    <li>
      <a href="$urlControl" data-container="body" data-bs-toggle="tooltip" data-placement="top" title="Control">
        <svg class="feather feather-clipboard" xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
          <path d="M16 4h2a2 2 0 0 1 2 2v14a2 2 0 0 1-2 2H6a2 2 0 0 1-2-2V6a2 2 0 0 1 2-2h2" />
          <rect x="8" y="2" width="8" height="4" rx="1" ry="1" />
        </svg>
        <span class="visually-hidden">Control</span>
      </a>
    </li>
    <li>
      <a href="$urlReportes" data-container="body" data-bs-toggle="tooltip" data-placement="top" title="Reportes">
        <svg class="feather feather-bar-chart-2" xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
          <line x1="18" y1="20" x2="18" y2="10" />
          <line x1="12" y1="20" x2="12" y2="4" />
          <line x1="6" y1="20" x2="6" y2="14" />
        </svg>
        <span class="visually-hidden">Reportes</span>
      </a>
    </li>
    <li>
      <a href="$urlParametros" data-container="body" data-bs-toggle="tooltip" data-placement="top" title="Parámetros">
        <svg class="feather feather-settings" xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
          <circle cx="12" cy="12" r="3" />
          <path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0l.06.06a1.65 1.65 0 0 0 1.82.33H9a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83l-.06.06a1.65 1.65 0 0 0-.33 1.82V9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09a1.65 1.65 0 0 0-1.51 1z" />
        </svg>
        <span class="visually-hidden">Parámetros</span>
      </a>
    </li>
  </ul>
</div>
<!-- Bookmark Ends-->
HTML;

$pageToolbar    = $GLOBALS['__sol_sections']['page_toolbar'] ?? $defaultToolbar;
$pageHeaderRaw  = $GLOBALS['__sol_sections']['page_header'] ?? null;
$pageHeaderHtml = '';

if (is_callable($pageHeaderRaw)) {
  ob_start();
  $pageHeaderRaw();
  $pageHeaderHtml = ob_get_clean();
} elseif (is_array($pageHeaderRaw)) {
  $pageHeaderHtml = implode("\n", array_map(static fn($v) => (string)$v, $pageHeaderRaw));
} elseif (is_string($pageHeaderRaw)) {
  $pageHeaderHtml = $pageHeaderRaw;
}

if ($pageHeaderHtml === '' || $pageHeaderHtml === null) {
  $safePageTitle = htmlspecialchars((string) $pageTitle, ENT_QUOTES, 'UTF-8');
  $pageHeaderHtml = <<<HTML
<div class="d-flex align-items-center gap-2">
  <h5 class="m-0">$safePageTitle</h5>
</div>
HTML;
}

$pageCss        = $GLOBALS['__sol_sections']['page_css'] ?? [];
$pageJs         = $GLOBALS['__sol_sections']['page_js'] ?? [];
$pageScriptsEnd = $GLOBALS['__sol_sections']['page_scripts_end'] ?? '';

// Slot de contenido
$renderBody     = $renderBody ?? null;
$content        = isset($content) ? (string)$content : null;
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <?php
    $__pageTitle    = $pageTitle;
    $__pageExtraCss = $pageCss;
    include project_path('views/layout/_partials/header.php');
  ?>
</head>
<body class="dark-sidebar" data-bs-theme="light">
  <!-- Loader opcional -->
  <div class="loader-wrapper" style="display:none;">
    <div class="theme-loader"><div class="loader-p"></div></div>
  </div>

  <!-- Page wrapper -->
  <div class="page-wrapper" id="pageWrapper">

    <!-- Topbar -->
    <?php include project_path('views/layout/_partials/topbar.php'); ?>
    <!-- /Topbar -->

    <!-- Body wrapper -->
    <div class="page-body-wrapper">
      <!-- Sidebar -->
      <?php include project_path('views/layout/_partials/sidebar.php'); ?>
      <!-- /Sidebar -->

      <!-- Page content -->
      <div class="page-body">
        <div class="container-fluid">
          <!-- Page header -->
          <div class="page-header">
            <div class="row align-items-center g-3">
              <div class="col-sm-6 col-12">
                <?= $pageHeaderHtml ?>
              </div>
              <div class="col-sm-6 col-12 text-sm-end">
                <?= $pageToolbar ?>
              </div>
            </div>
          </div>
          <!-- /Page header -->

          <!-- Main slot -->
          <main>
            <?php
              if (is_callable($renderBody)) {
                $renderBody();
              } elseif (is_string($content)) {
                echo $content;
              } else {
                echo '<div class="alert alert-info">Sin contenido. Integra este layout incluyendo $content o $renderBody desde tu vista.</div>';
              }
            ?>
          </main>
          <!-- /Main slot -->
        </div>
      </div>
      <!-- /Page content -->

      <!-- Footer -->
      <footer class="footer">
        <div class="container-fluid">
          <div class="row">
            <div class="col-md-6 footer-copyright">
              <p class="mb-0">&copy; <?= date('Y') ?> <?= htmlspecialchars($appName, ENT_QUOTES, 'UTF-8') ?> - Todos los derechos reservados.</p>
            </div>
            <div class="col-md-6 text-md-end">
              <p class="mb-0">Hecho con <i class="fa fa-heart font-secondary"></i> by arasaIT</p>
            </div>
          </div>
        </div>
      </footer>
      <!-- /Footer -->
    </div>
    <!-- /Body wrapper -->
  </div>
  <!-- /Page wrapper -->

  <?php
    // Footer scripts
    include project_path('views/layout/_partials/footer.php');
    $pageJs         = is_array($pageJs) ? $pageJs : [];
    $pageScriptsEnd = is_string($pageScriptsEnd) ? $pageScriptsEnd : '';
  ?>
</body>
</html>
