<?php

declare(strict_types=1);

if (!headers_sent()) {
    header('Content-Type: application/pdf; charset=utf-8');
}

$BASE = dirname(__DIR__, 3);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';
require_once $BASE . '/app/Support/Reportes/IngresosReport.php';

if (empty($_SESSION['usuario_id'])) {
    http_response_code(401);
    echo 'No autorizado';
    exit;
}

$limit = isset($_GET['limit']) ? (int) $_GET['limit'] : 1500;
$limit = $limit > 0 ? $limit : 1500;

$pdo = getPDO();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

$data = ingresos_fetch_data($pdo, $_GET, $limit);
$combos = ingresos_fetch_combos($pdo);

$filters = $data['filters'];
$summary = $data['summary'];
$rows = $data['rows'];
$truncated = !empty($data['truncated']);
$limitUsed = (int) ($data['limit'] ?? $limit);

$clienteLabel = 'Todos';
if ($filters['cliente_id'] !== '') {
    foreach ($combos['clientes'] as $cliente) {
        if ((string) $cliente['id'] === (string) $filters['cliente_id']) {
            $clienteLabel = $cliente['nombre'];
            break;
        }
    }
}

$depositoLabel = 'Todos';
if ($filters['deposito_id'] !== '') {
    foreach ($combos['depositos'] as $deposito) {
        if ((string) $deposito['id'] === (string) $filters['deposito_id']) {
            $depositoLabel = $deposito['nombre'];
            if (!empty($deposito['code'])) {
                $depositoLabel .= ' [' . $deposito['code'] . ']';
            }
            break;
        }
    }
}

$estadoLabel = 'Todos';
if ($filters['estado'] !== '') {
    foreach ($combos['estados'] as $estado) {
        if ((string) $estado['code'] === (string) $filters['estado']) {
            $estadoLabel = $estado['nombre'] . ' [' . $estado['code'] . ']';
            break;
        }
    }
}

$movilLabel = 'Todos';
if ($filters['movil_id'] !== '') {
    foreach ($combos['moviles'] as $movil) {
        if ((string) $movil['id'] === (string) $filters['movil_id']) {
            $movilLabel = $movil['label'];
            break;
        }
    }
}

$codigoLabel = $filters['codigo'] !== '' ? $filters['codigo'] : 'Todos';
$docLabel = $filters['doc_numero'] !== '' ? $filters['doc_numero'] : 'Todos';

$generatedAt = (new DateTimeImmutable('now'))->format('d/m/Y H:i');
$filename = 'reporte_ingresos_' . (new DateTimeImmutable('now'))->format('Ymd_His') . '.pdf';

function pr_num($value, int $decimals = 0): string
{
    $num = (float) $value;
    return number_format($num, $decimals, ',', '.');
}

ob_start();
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Reporte de Ingresos</title>
  <style>
    body { font-family: 'DejaVu Sans', sans-serif; font-size: 9.5pt; color: #222; }
    h1 { font-size: 16pt; margin-bottom: 6px; }
    .meta { font-size: 9pt; margin-bottom: 12px; }
    .meta span { display: inline-block; margin-right: 12px; }
    table { width: 100%; border-collapse: collapse; margin-top: 10px; }
    th, td { border: 1px solid #bbb; padding: 6px 4px; }
    th { background: #f2f2f2; font-weight: bold; text-align: center; }
    td { text-align: left; }
    td.num { text-align: right; }
    tfoot td { font-weight: bold; background: #f9f9f9; }
    .summary-box { margin-bottom: 10px; padding: 8px 10px; background: #f6f6f6; border: 1px solid #ddd; border-radius: 4px; }
    .summary-box strong { display: inline-block; min-width: 160px; }
    .warning { color: #c0392b; }
  </style>
</head>
<body>
  <h1>Reporte de Ingresos</h1>
  <p class="meta">
    <span><strong>Generado:</strong> <?= htmlspecialchars($generatedAt, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Rango:</strong> <?= htmlspecialchars($filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'], ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Cliente:</strong> <?= htmlspecialchars($clienteLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Depósito:</strong> <?= htmlspecialchars($depositoLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Estado:</strong> <?= htmlspecialchars($estadoLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Móvil:</strong> <?= htmlspecialchars($movilLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Packing List:</strong> <?= htmlspecialchars($codigoLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Documento:</strong> <?= htmlspecialchars($docLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Registros:</strong> <?= pr_num($summary['total_ingresos'] ?? 0) ?></span>
  </p>

  <div class="summary-box">
    <div><strong>Clientes únicos:</strong> <?= pr_num($summary['total_clientes'] ?? 0) ?></div>
    <div><strong>Depósitos únicos:</strong> <?= pr_num($summary['total_depositos'] ?? 0) ?> · <strong>Móviles únicos:</strong> <?= pr_num($summary['total_moviles'] ?? 0) ?></div>
    <div><strong>Pallets recibidos:</strong> <?= pr_num($summary['total_pallets'] ?? 0) ?></div>
    <div><strong>Unidades UV:</strong> <?= pr_num($summary['total_uv'] ?? 0) ?> · <strong>Unidades UC:</strong> <?= pr_num($summary['total_uc'] ?? 0) ?></div>
  </div>

  <table>
    <thead>
      <tr>
        <th>#</th>
        <th>Packing List</th>
        <th>Fecha</th>
        <th>Cliente</th>
        <th>Depósito</th>
        <th>Estado</th>
        <th>Documento</th>
        <th>Pallets</th>
        <th>UV</th>
        <th>UC</th>
        <th>Ítems</th>
        <th>Operarios</th>
        <th>Camión</th>
        <th>Chofer</th>
      </tr>
    </thead>
    <tbody>
      <?php if (empty($rows)): ?>
        <tr>
          <td colspan="14" style="text-align:center; padding:16px;">Sin datos que coincidan con los filtros seleccionados.</td>
        </tr>
      <?php else: ?>
        <?php foreach ($rows as $index => $row): ?>
          <tr>
            <td class="num"><?= $index + 1 ?></td>
            <td><?= htmlspecialchars($row['packinglist_codigo'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars(($row['fecha_ingreso'] ?? '-') . ($row['hora_llegada'] ? ' ' . $row['hora_llegada'] : ''), ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($row['cliente_nombre'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($row['deposito_nombre'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($row['estado_nombre'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($row['doc_referencia'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td class="num"><?= pr_num($row['pallets'] ?? 0) ?></td>
            <td class="num"><?= pr_num($row['uv_cajas'] ?? 0) ?></td>
            <td class="num"><?= pr_num($row['uc_unidades'] ?? 0) ?></td>
            <td class="num"><?= pr_num($row['items'] ?? 0) ?></td>
            <td class="num"><?= pr_num($row['operarios_cant'] ?? 0) ?></td>
            <td><?= htmlspecialchars($row['movil_chapa'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($row['chofer_nombre'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
          </tr>
        <?php endforeach; ?>
      <?php endif; ?>
    </tbody>
    <tfoot>
      <tr>
        <td colspan="7">Totales</td>
        <td class="num"><?= pr_num($summary['total_pallets'] ?? 0) ?></td>
        <td class="num"><?= pr_num($summary['total_uv'] ?? 0) ?></td>
        <td class="num"><?= pr_num($summary['total_uc'] ?? 0) ?></td>
        <td class="num"><?= pr_num(array_sum(array_column($rows, 'items')) ?: 0) ?></td>
        <td class="num"><?= pr_num(array_sum(array_map(static fn($r) => (int) ($r['operarios_cant'] ?? 0), $rows))) ?></td>
        <td colspan="2"></td>
      </tr>
    </tfoot>
  </table>

  <?php if ($truncated): ?>
    <p class="warning" style="margin-top:10px;">
      Se muestran hasta <?= pr_num($limitUsed) ?> registros. Refine los filtros para obtener mayor detalle.
    </p>
  <?php endif; ?>
</body>
</html>
<?php
$html = ob_get_clean();

$mpdf = new \Mpdf\Mpdf([
    'mode' => 'utf-8',
    'format' => 'A4-L',
    'margin_left' => 10,
    'margin_right' => 10,
    'margin_top' => 12,
    'margin_bottom' => 12,
]);
$mpdf->SetTitle('Reporte de Ingresos');
$mpdf->SetAuthor((string) env('APP_NAME', 'SOL'));
$mpdf->WriteHTML($html);
$mpdf->Output($filename, 'I');
exit;
