<?php

declare(strict_types=1);

if (!headers_sent()) {
    header('Content-Type: application/pdf; charset=utf-8');
}

$BASE = dirname(__DIR__, 3);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';
require_once $BASE . '/app/Support/Reportes/MovilesDisponiblesReport.php';

if (empty($_SESSION['usuario_id'])) {
    http_response_code(401);
    echo 'No autorizado';
    exit;
}

$limit = isset($_GET['limit']) ? (int) $_GET['limit'] : 1500;
$limit = $limit > 0 ? $limit : 1500;

$pdo = getPDO();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

$data = moviles_disponibles_fetch_data($pdo, $_GET, $limit);
$combos = moviles_disponibles_fetch_combos($pdo);

$filters = $data['filters'];
$summary = $data['summary'];
$rows = $data['rows'];
$truncated = !empty($data['truncated']);
$limitUsed = (int) ($data['limit'] ?? $limit);

$transportadoraLabel = 'Todas';
if ($filters['transportadora_id'] !== '') {
    foreach ($combos['transportadoras'] as $transportadora) {
        if ((string) $transportadora['id'] === (string) $filters['transportadora_id']) {
            $transportadoraLabel = $transportadora['nombre'];
            break;
        }
    }
}

$movilLabel = 'Todos';
if ($filters['movil_id'] !== '') {
    foreach ($combos['moviles'] as $movil) {
        if ((string) $movil['id'] === (string) $filters['movil_id']) {
            $movilLabel = $movil['label'];
            break;
        }
    }
}

function md_label_from_combo(array $list, string $value, string $fallback): string
{
    foreach ($list as $item) {
        if ((string) $item['value'] === $value) {
            return $item['label'];
        }
    }
    return $fallback;
}

$disponibleLabel = md_label_from_combo($combos['estados'], $filters['disponible'], 'Todos');
$noAsistioLabel = md_label_from_combo($combos['binarios'], $filters['no_asistio'], 'Todos');
$noUtilizadoLabel = md_label_from_combo($combos['binarios'], $filters['no_utilizado'], 'Todos');
$buscarLabel = $filters['buscar'] !== '' ? $filters['buscar'] : 'Todos';

$generatedAt = (new DateTimeImmutable('now'))->format('d/m/Y H:i');
$filename = 'reporte_moviles_disponibles_' . (new DateTimeImmutable('now'))->format('Ymd_His') . '.pdf';

function md_num($value, int $decimals = 0): string
{
    $num = (float) $value;
    return number_format($num, $decimals, ',', '.');
}

ob_start();
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Reporte de Móviles Disponibles</title>
  <style>
    body { font-family: 'DejaVu Sans', sans-serif; font-size: 9.5pt; color: #222; }
    h1 { font-size: 16pt; margin-bottom: 6px; }
    .meta { font-size: 9pt; margin-bottom: 12px; }
    .meta span { display: inline-block; margin-right: 12px; }
    table { width: 100%; border-collapse: collapse; margin-top: 10px; }
    th, td { border: 1px solid #bbb; padding: 6px 4px; }
    th { background: #f2f2f2; font-weight: bold; text-align: center; }
    td { text-align: left; }
    td.num { text-align: right; }
    tfoot td { font-weight: bold; background: #f9f9f9; }
    .summary-box { margin-bottom: 10px; padding: 8px 10px; background: #f6f6f6; border: 1px solid #ddd; border-radius: 4px; }
    .summary-box strong { display: inline-block; min-width: 170px; }
    .warning { color: #c0392b; }
  </style>
</head>
<body>
  <h1>Reporte de Móviles Disponibles</h1>
  <p class="meta">
    <span><strong>Generado:</strong> <?= htmlspecialchars($generatedAt, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Rango:</strong> <?= htmlspecialchars($filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'], ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Transportadora:</strong> <?= htmlspecialchars($transportadoraLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Móvil:</strong> <?= htmlspecialchars($movilLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Disponibilidad:</strong> <?= htmlspecialchars($disponibleLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>No asistió:</strong> <?= htmlspecialchars($noAsistioLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>No utilizado:</strong> <?= htmlspecialchars($noUtilizadoLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Buscar:</strong> <?= htmlspecialchars($buscarLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Registros:</strong> <?= md_num($summary['total_registros'] ?? 0) ?></span>
  </p>

  <div class="summary-box">
    <div><strong>Móviles únicos:</strong> <?= md_num($summary['total_moviles'] ?? 0) ?> · <strong>Transportadoras únicas:</strong> <?= md_num($summary['total_transportadoras'] ?? 0) ?></div>
    <div><strong>Disponibles:</strong> <?= md_num($summary['total_disponibles'] ?? 0) ?> · <strong>No disponibles:</strong> <?= md_num($summary['total_no_disponibles'] ?? 0) ?></div>
    <div><strong>No asistió:</strong> <?= md_num($summary['total_no_asistio'] ?? 0) ?> · <strong>No utilizado:</strong> <?= md_num($summary['total_no_utilizado'] ?? 0) ?></div>
  </div>

  <table>
    <thead>
      <tr>
        <th>#</th>
        <th>Fecha</th>
        <th>Móvil</th>
        <th>Transportadora</th>
        <th>Disponible</th>
        <th>Ventana</th>
        <th>Llegada</th>
        <th>No asistió</th>
        <th>No utilizado</th>
        <th>Notas</th>
      </tr>
    </thead>
    <tbody>
      <?php if (empty($rows)): ?>
        <tr>
          <td colspan="10" style="text-align:center; padding:16px;">Sin datos que coincidan con los filtros seleccionados.</td>
        </tr>
      <?php else: ?>
        <?php foreach ($rows as $index => $row): ?>
          <tr>
            <td class="num"><?= $index + 1 ?></td>
            <td><?= htmlspecialchars($row['fecha'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($row['movil_label'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($row['transportadora_nombre'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= !empty($row['disponible']) ? 'Sí' : 'No' ?></td>
            <td><?= htmlspecialchars(trim(($row['ventana_ini'] ?? '-') . ' - ' . ($row['ventana_fin'] ?? '-')), ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($row['llegada_time'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= !empty($row['no_asistio']) ? 'Sí' : 'No' ?></td>
            <td><?= !empty($row['no_utilizado']) ? 'Sí' : 'No' ?></td>
            <td><?= htmlspecialchars($row['notas'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
          </tr>
        <?php endforeach; ?>
      <?php endif; ?>
    </tbody>
  </table>

  <?php if ($truncated): ?>
    <p class="warning" style="margin-top:10px;">
      Se muestran hasta <?= md_num($limitUsed) ?> registros. Refine los filtros para obtener mayor detalle.
    </p>
  <?php endif; ?>
</body>
</html>
<?php
$html = ob_get_clean();

$mpdf = new \Mpdf\Mpdf([
    'mode' => 'utf-8',
    'format' => 'A4-L',
    'margin_left' => 10,
    'margin_right' => 10,
    'margin_top' => 12,
    'margin_bottom' => 12,
]);
$mpdf->SetTitle('Reporte de Móviles Disponibles');
$mpdf->SetAuthor((string) env('APP_NAME', 'SOL'));
$mpdf->WriteHTML($html);
$mpdf->Output($filename, 'I');
exit;
