<?php

declare(strict_types=1);

if (!headers_sent()) {
    header('Content-Type: application/pdf; charset=utf-8');
}

$BASE = dirname(__DIR__, 3);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';
require_once $BASE . '/app/Support/Reportes/PedidosReport.php';

if (empty($_SESSION['usuario_id'])) {
    http_response_code(401);
    echo 'No autorizado';
    exit;
}

$limit = isset($_GET['limit']) ? (int) $_GET['limit'] : 1500;
$limit = $limit > 0 ? $limit : 1500;

$pdo = getPDO();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

$data = pedidos_fetch_data($pdo, $_GET, $limit);
$combos = pedidos_fetch_combos($pdo);

$filters = $data['filters'];
$summary = $data['summary'];
$rows = $data['rows'];
$truncated = !empty($data['truncated']);
$limitUsed = (int) ($data['limit'] ?? $limit);

$clienteLabel = 'Todos';
if ($filters['cliente_id'] !== '') {
    $clienteLabel = 'Cliente #' . $filters['cliente_id'];
    foreach ($combos['clientes'] as $cliente) {
        if ((string) $cliente['id'] === (string) $filters['cliente_id']) {
            $clienteLabel = $cliente['nombre'];
            break;
        }
    }
}

$estadoLabel = 'Todos';
if ($filters['estado_id'] !== '') {
    $estadoLabel = 'Estado #' . $filters['estado_id'];
    foreach ($combos['estados'] as $estado) {
        if ((string) $estado['id'] === (string) $filters['estado_id']) {
            $estadoLabel = $estado['nombre'];
            if (!empty($estado['code'])) {
                $estadoLabel .= ' [' . $estado['code'] . ']';
            }
            break;
        }
    }
}

$codigoLabel = $filters['codigo'] !== '' ? $filters['codigo'] : 'Todos';

$generatedAt = (new DateTimeImmutable('now'))->format('d/m/Y H:i');
$filename = 'reporte_pedidos_' . (new DateTimeImmutable('now'))->format('Ymd_His') . '.pdf';

function pr_num($value, int $decimals = 0): string
{
    $num = (float) $value;
    return number_format($num, $decimals, ',', '.');
}

function pr_percent($value): string
{
    if ($value === null) {
        return '-';
    }
    $num = (float) $value;
    return number_format($num, 1, ',', '.') . '%';
}

ob_start();
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Reporte de Pedidos</title>
  <style>
    body { font-family: 'DejaVu Sans', sans-serif; font-size: 9.5pt; color: #222; }
    h1 { font-size: 16pt; margin-bottom: 6px; }
    .meta { font-size: 9pt; margin-bottom: 12px; }
    .meta span { display: inline-block; margin-right: 12px; }
    table { width: 100%; border-collapse: collapse; margin-top: 10px; }
    th, td { border: 1px solid #bbb; padding: 6px 4px; }
    th { background: #f2f2f2; font-weight: bold; text-align: center; }
    td { text-align: left; }
    td.num { text-align: right; }
    tfoot td { font-weight: bold; background: #f9f9f9; }
    .summary-box { margin-bottom: 10px; padding: 8px 10px; background: #f6f6f6; border: 1px solid #ddd; border-radius: 4px; }
    .summary-box strong { display: inline-block; min-width: 140px; }
    .warning { color: #c0392b; }
  </style>
</head>
<body>
  <h1>Reporte de Pedidos</h1>
  <p class="meta">
    <span><strong>Generado:</strong> <?= htmlspecialchars($generatedAt, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Rango:</strong> <?= htmlspecialchars($filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'], ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Cliente:</strong> <?= htmlspecialchars($clienteLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Estado:</strong> <?= htmlspecialchars($estadoLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Código:</strong> <?= htmlspecialchars($codigoLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Registros:</strong> <?= pr_num($summary['total_pedidos'] ?? 0) ?></span>
  </p>

  <div class="summary-box">
    <div><strong>Clientes únicos:</strong> <?= pr_num($summary['total_clientes'] ?? 0) ?></div>
    <div><strong>Destinos:</strong> <?= pr_num($summary['total_destinos'] ?? 0) ?> · <strong>Destinatarios:</strong> <?= pr_num($summary['total_destinatarios'] ?? 0) ?></div>
    <div><strong>Unidades UC:</strong> <?= pr_num($summary['total_expected_uc'] ?? 0) ?> planificadas · <?= pr_num($summary['total_shipped_uc'] ?? 0) ?> despachadas</div>
    <div><strong>Unidades UV:</strong> <?= pr_num($summary['total_expected_uv'] ?? 0) ?> planificadas · <?= pr_num($summary['total_shipped_uv'] ?? 0) ?> despachadas</div>
  </div>

  <table>
    <thead>
      <tr>
        <th>#</th>
        <th>Código</th>
        <th>Fecha</th>
        <th>Cliente</th>
        <th>Estado</th>
        <th>Destinos</th>
        <th>Destinatarios</th>
        <th>Ítems</th>
        <th>UV (Plan)</th>
        <th>UV (Desp)</th>
        <th>UC (Plan)</th>
        <th>UC (Desp)</th>
        <th>Avance UC</th>
      </tr>
    </thead>
    <tbody>
      <?php if (empty($rows)): ?>
        <tr>
          <td colspan="13" style="text-align:center; padding:16px;">Sin datos que coincidan con los filtros seleccionados.</td>
        </tr>
      <?php else: ?>
        <?php foreach ($rows as $index => $row): ?>
          <tr>
            <td class="num"><?= $index + 1 ?></td>
            <td><?= htmlspecialchars($row['pedido_codigo'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars(($row['fecha_pedido'] ?? '-') . ($row['hora_pedido'] ? ' ' . $row['hora_pedido'] : ''), ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($row['cliente_nombre'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($row['estado_nombre'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td class="num"><?= pr_num($row['destinos'] ?? 0) ?></td>
            <td class="num"><?= pr_num($row['destinatarios'] ?? 0) ?></td>
            <td class="num"><?= pr_num($row['items'] ?? 0) ?></td>
            <td class="num"><?= pr_num($row['expected_uv'] ?? 0) ?></td>
            <td class="num"><?= pr_num($row['shipped_uv'] ?? 0) ?></td>
            <td class="num"><?= pr_num($row['expected_uc'] ?? 0) ?></td>
            <td class="num"><?= pr_num($row['shipped_uc'] ?? 0) ?></td>
            <td class="num"><?= pr_percent($row['avance_uc_pct'] ?? null) ?></td>
          </tr>
        <?php endforeach; ?>
      <?php endif; ?>
    </tbody>
    <tfoot>
      <tr>
        <td colspan="5">Totales</td>
        <td class="num"><?= pr_num($summary['total_destinos'] ?? 0) ?></td>
        <td class="num"><?= pr_num($summary['total_destinatarios'] ?? 0) ?></td>
        <td class="num"><?= pr_num($summary['total_items'] ?? 0) ?></td>
        <td class="num"><?= pr_num($summary['total_expected_uv'] ?? 0) ?></td>
        <td class="num"><?= pr_num($summary['total_shipped_uv'] ?? 0) ?></td>
        <td class="num"><?= pr_num($summary['total_expected_uc'] ?? 0) ?></td>
        <td class="num"><?= pr_num($summary['total_shipped_uc'] ?? 0) ?></td>
        <td></td>
      </tr>
    </tfoot>
  </table>

  <?php if ($truncated): ?>
    <p class="warning" style="margin-top:10px;">
      Se muestran hasta <?= pr_num($limitUsed) ?> registros. Refine los filtros para obtener mayor detalle.
    </p>
  <?php endif; ?>
</body>
</html>
<?php
$html = ob_get_clean();

$mpdf = new \Mpdf\Mpdf([
    'mode' => 'utf-8',
    'format' => 'A4-L',
    'margin_left' => 10,
    'margin_right' => 10,
    'margin_top' => 12,
    'margin_bottom' => 12,
]);
$mpdf->SetTitle('Reporte de Pedidos');
$mpdf->SetAuthor((string) env('APP_NAME', 'SOL'));
$mpdf->WriteHTML($html);
$mpdf->Output($filename, 'I');
exit;
