<?php

declare(strict_types=1);

if (!headers_sent()) {
    header('Content-Type: application/pdf; charset=utf-8');
}

$BASE = dirname(__DIR__, 3);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';
require_once $BASE . '/app/Support/Reportes/PedidosHoraReport.php';

if (empty($_SESSION['usuario_id'])) {
    http_response_code(401);
    echo 'No autorizado';
    exit;
}

$limit = isset($_GET['limit']) ? (int) $_GET['limit'] : 90;
$limit = $limit > 0 ? $limit : 90;

$pdo = getPDO();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

$data = pedidos_hora_fetch_data($pdo, $_GET, $limit);
$summary = $data['summary'] ?? [];
$hours = $data['hours'] ?? [];
$daily = $data['daily'] ?? [];
$aggregates = $data['aggregates'] ?? ['clientes' => [], 'estados' => []];
$filters = $data['filters'] ?? [];
$limitDays = (int) ($data['limit_days'] ?? $limit);

$generatedAt = (new DateTimeImmutable('now'))->format('d/m/Y H:i');
$filename = 'reporte_pedidos_por_hora_' . (new DateTimeImmutable('now'))->format('Ymd_His') . '.pdf';

function ph_num($value, int $decimals = 0): string
{
    $num = (float) $value;
    return number_format($num, $decimals, ',', '.');
}

function ph_time_label(?string $value): string
{
    if ($value === null || $value === '') {
        return '-';
    }
    return substr($value, 0, 5);
}

function ph_date_label(?string $value): string
{
    if ($value === null || $value === '') {
        return '-';
    }
    $parts = explode('-', $value);
    if (count($parts) !== 3) {
        return $value;
    }
    return $parts[2] . '/' . $parts[1] . '/' . $parts[0];
}

ob_start();
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Reporte de Pedidos por Hora</title>
  <style>
    body { font-family: 'DejaVu Sans', sans-serif; font-size: 10pt; color: #222; }
    h1 { font-size: 18pt; margin-bottom: 4px; }
    h2 { font-size: 14pt; margin-top: 20px; }
    .meta { font-size: 9pt; margin-bottom: 10px; }
    .meta span { display: inline-block; margin-right: 12px; }
    table { width: 100%; border-collapse: collapse; margin-top: 10px; }
    th, td { border: 1px solid #bbb; padding: 6px 4px; }
    th { background: #f2f2f2; text-align: center; font-weight: bold; }
    td.text-end { text-align: right; }
    ul { list-style: none; padding: 0; margin: 0; }
    ul li { padding: 6px 4px; border-bottom: 1px solid #ddd; display: flex; justify-content: space-between; }
    .badge { background: #0d6efd; color: #fff; padding: 2px 8px; border-radius: 12px; font-size: 9pt; }
    .summary-grid { display: grid; grid-template-columns: repeat(2, 1fr); gap: 6px 16px; margin-top: 10px; }
    .summary-grid span { display: inline-block; }
    .warning { color: #c0392b; margin-top: 6px; }
  </style>
</head>
<body>
  <h1>Reporte de Pedidos por Hora</h1>
  <p class="meta">
    <span><strong>Generado:</strong> <?= htmlspecialchars($generatedAt, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Rango:</strong> <?= htmlspecialchars($summary['range_label'] ?? '', ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Total pedidos:</strong> <?= ph_num($summary['total_pedidos'] ?? 0) ?></span>
    <span><strong>Clientes únicos:</strong> <?= ph_num($summary['total_clientes'] ?? 0) ?></span>
    <span><strong>Días analizados:</strong> <?= ph_num($summary['total_dias'] ?? 0) ?></span>
  </p>

  <div class="summary-grid">
    <span><strong>Promedio por día:</strong> <?= ph_num($summary['promedio_diario'] ?? 0, 2) ?></span>
    <span><strong>Promedio por hora:</strong> <?= ph_num($summary['promedio_por_hora'] ?? 0, 2) ?></span>
    <span><strong>Horas activas:</strong> <?= ph_num($summary['horas_activas'] ?? 0) ?></span>
    <span><strong>Promedio por hora activa:</strong> <?= ph_num($summary['promedio_por_hora_activa'] ?? 0, 2) ?></span>
    <span><strong>Hora pico:</strong> <?= htmlspecialchars($summary['hora_pico'] ?? '-', ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Pedidos hora pico:</strong> <?= ph_num($summary['hora_pico_pedidos'] ?? 0) ?></span>
    <span><strong>Primer registro:</strong> <?= htmlspecialchars((string) ($summary['primer_registro'] ?? '-'), ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Último registro:</strong> <?= htmlspecialchars((string) ($summary['ultimo_registro'] ?? '-'), ENT_QUOTES, 'UTF-8') ?></span>
  </div>

  <h2>Distribución por hora</h2>
  <table>
    <thead>
      <tr>
        <th>Hora</th>
        <th>Pedidos</th>
        <th>Promedio por día</th>
        <th>Clientes</th>
        <th>% del total</th>
      </tr>
    </thead>
    <tbody>
      <?php if (empty($hours)): ?>
        <tr><td colspan="5" style="text-align:center; padding:12px;">Sin datos disponibles</td></tr>
      <?php else: ?>
        <?php foreach ($hours as $row): ?>
          <tr>
            <td><?= htmlspecialchars($row['label'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td class="text-end"><?= ph_num($row['pedidos'] ?? 0) ?></td>
            <td class="text-end"><?= ph_num($row['promedio_por_dia'] ?? 0, 2) ?></td>
            <td class="text-end"><?= ph_num($row['clientes'] ?? 0) ?></td>
            <td class="text-end"><?= ph_num($row['porcentaje'] ?? 0, 2) ?>%</td>
          </tr>
        <?php endforeach; ?>
      <?php endif; ?>
    </tbody>
  </table>

  <h2>Detalle diario (máx. <?= ph_num($limitDays) ?> días)</h2>
  <table>
    <thead>
      <tr>
        <th>Fecha</th>
        <th>Pedidos</th>
        <th>Clientes</th>
        <th>Primer pedido</th>
        <th>Último pedido</th>
      </tr>
    </thead>
    <tbody>
      <?php if (empty($daily)): ?>
        <tr><td colspan="5" style="text-align:center; padding:12px;">Sin datos disponibles</td></tr>
      <?php else: ?>
        <?php foreach ($daily as $row): ?>
          <tr>
            <td><?= htmlspecialchars(ph_date_label($row['fecha'] ?? null), ENT_QUOTES, 'UTF-8') ?></td>
            <td class="text-end"><?= ph_num($row['pedidos'] ?? 0) ?></td>
            <td class="text-end"><?= ph_num($row['clientes'] ?? 0) ?></td>
            <td class="text-end"><?= htmlspecialchars(ph_time_label($row['primer_pedido'] ?? null), ENT_QUOTES, 'UTF-8') ?></td>
            <td class="text-end"><?= htmlspecialchars(ph_time_label($row['ultimo_pedido'] ?? null), ENT_QUOTES, 'UTF-8') ?></td>
          </tr>
        <?php endforeach; ?>
      <?php endif; ?>
    </tbody>
  </table>

  <h2>Top clientes</h2>
  <ul>
    <?php if (empty($aggregates['clientes'])): ?>
      <li style="text-align:center; color:#777;">Sin información disponible</li>
    <?php else: ?>
      <?php foreach ($aggregates['clientes'] as $cliente): ?>
        <li>
          <span><?= htmlspecialchars($cliente['nombre'] ?? 'Sin cliente', ENT_QUOTES, 'UTF-8') ?></span>
          <span class="badge"><?= ph_num($cliente['pedidos'] ?? 0) ?></span>
        </li>
      <?php endforeach; ?>
    <?php endif; ?>
  </ul>

  <h2>Estados</h2>
  <ul>
    <?php if (empty($aggregates['estados'])): ?>
      <li style="text-align:center; color:#777;">Sin información disponible</li>
    <?php else: ?>
      <?php foreach ($aggregates['estados'] as $estado): ?>
        <li>
          <span><?= htmlspecialchars(trim(($estado['code'] ?? '') . ' ' . ($estado['nombre'] ?? 'Sin estado')), ENT_QUOTES, 'UTF-8') ?></span>
          <span class="badge"><?= ph_num($estado['pedidos'] ?? 0) ?></span>
        </li>
      <?php endforeach; ?>
    <?php endif; ?>
  </ul>

  <?php if (!empty($daily) && count($daily) >= $limitDays): ?>
    <p class="warning">Se alcanzó el límite de <?= ph_num($limitDays) ?> días. Refine el rango para obtener más detalle.</p>
  <?php endif; ?>
</body>
</html>
<?php
$html = ob_get_clean();

$mpdf = new \Mpdf\Mpdf([
    'mode' => 'utf-8',
    'format' => 'A4-L',
    'margin_left' => 12,
    'margin_right' => 12,
    'margin_top' => 12,
    'margin_bottom' => 12,
]);
$mpdf->SetTitle('Reporte de Pedidos por Hora');
$mpdf->SetAuthor((string) env('APP_NAME', 'SOL'));
$mpdf->WriteHTML($html);
$mpdf->Output($filename, 'I');
exit;
