<?php

declare(strict_types=1);

if (!headers_sent()) {
    header('Content-Type: application/pdf; charset=utf-8');
}

$BASE = dirname(__DIR__, 3);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';
require_once $BASE . '/app/Support/Reportes/SalidasReport.php';

if (empty($_SESSION['usuario_id'])) {
    http_response_code(401);
    echo 'No autorizado';
    exit;
}

$limit = isset($_GET['limit']) ? (int) $_GET['limit'] : 1500;
$limit = $limit > 0 ? $limit : 1500;

$pdo = getPDO();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

$data = salidas_fetch_data($pdo, $_GET, $limit);
$combos = salidas_fetch_combos($pdo);

$filters = $data['filters'];
$summary = $data['summary'];
$rows = $data['rows'];
$truncated = !empty($data['truncated']);
$limitUsed = (int) ($data['limit'] ?? $limit);

$depositoLabel = 'Todos';
if ($filters['deposito_id'] !== '') {
    foreach ($combos['depositos'] as $deposito) {
        if ((string) $deposito['id'] === (string) $filters['deposito_id']) {
            $depositoLabel = $deposito['nombre'];
            if (!empty($deposito['code'])) {
                $depositoLabel .= ' [' . $deposito['code'] . ']';
            }
            break;
        }
    }
}

$estadoLabel = 'Todos';
if ($filters['estado'] !== '') {
    foreach ($combos['estados'] as $estado) {
        if ((string) $estado['code'] === (string) $filters['estado']) {
            $estadoLabel = $estado['nombre'] . ' [' . $estado['code'] . ']';
            break;
        }
    }
}

$movilLabel = 'Todos';
if ($filters['movil_id'] !== '') {
    foreach ($combos['moviles'] as $movil) {
        if ((string) $movil['id'] === (string) $filters['movil_id']) {
            $movilLabel = $movil['label'];
            break;
        }
    }
}

$choferLabel = 'Todos';
if ($filters['chofer_id'] !== '') {
    foreach ($combos['choferes'] as $chofer) {
        if ((string) $chofer['id'] === (string) $filters['chofer_id']) {
            $choferLabel = $chofer['nombre'];
            break;
        }
    }
}

$codigoLabel = $filters['codigo'] !== '' ? $filters['codigo'] : 'Todos';

$generatedAt = (new DateTimeImmutable('now'))->format('d/m/Y H:i');
$filename = 'reporte_salidas_' . (new DateTimeImmutable('now'))->format('Ymd_His') . '.pdf';

function pr_num($value, int $decimals = 0): string
{
    $num = (float) $value;
    return number_format($num, $decimals, ',', '.');
}

ob_start();
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="utf-8">
  <title>Reporte de Salidas</title>
  <style>
    body { font-family: 'DejaVu Sans', sans-serif; font-size: 9.5pt; color: #222; }
    h1 { font-size: 16pt; margin-bottom: 6px; }
    .meta { font-size: 9pt; margin-bottom: 12px; }
    .meta span { display: inline-block; margin-right: 12px; }
    table { width: 100%; border-collapse: collapse; margin-top: 10px; }
    th, td { border: 1px solid #bbb; padding: 6px 4px; }
    th { background: #f2f2f2; font-weight: bold; text-align: center; }
    td { text-align: left; }
    td.num { text-align: right; }
    tfoot td { font-weight: bold; background: #f9f9f9; }
    .summary-box { margin-bottom: 10px; padding: 8px 10px; background: #f6f6f6; border: 1px solid #ddd; border-radius: 4px; }
    .summary-box strong { display: inline-block; min-width: 170px; }
    .warning { color: #c0392b; }
  </style>
</head>
<body>
  <h1>Reporte de Salidas</h1>
  <p class="meta">
    <span><strong>Generado:</strong> <?= htmlspecialchars($generatedAt, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Rango:</strong> <?= htmlspecialchars($filters['fecha_desde'] . ' al ' . $filters['fecha_hasta'], ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Depósito:</strong> <?= htmlspecialchars($depositoLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Estado:</strong> <?= htmlspecialchars($estadoLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Móvil:</strong> <?= htmlspecialchars($movilLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Chofer:</strong> <?= htmlspecialchars($choferLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Código:</strong> <?= htmlspecialchars($codigoLabel, ENT_QUOTES, 'UTF-8') ?></span>
    <span><strong>Registros:</strong> <?= pr_num($summary['total_salidas'] ?? 0) ?></span>
  </p>

  <div class="summary-box">
    <div><strong>Depósitos únicos:</strong> <?= pr_num($summary['total_depositos'] ?? 0) ?> · <strong>Móviles únicos:</strong> <?= pr_num($summary['total_moviles'] ?? 0) ?> · <strong>Choferes únicos:</strong> <?= pr_num($summary['total_choferes'] ?? 0) ?></div>
    <div><strong>Total destinos:</strong> <?= pr_num($summary['total_destinos'] ?? 0) ?> · <strong>Total preembarques:</strong> <?= pr_num($summary['total_preembarques'] ?? 0) ?></div>
    <div><strong>Promedio espera:</strong> <?= pr_num($summary['avg_espera_min'] ?? 0, 1) ?> min · <strong>Promedio carga:</strong> <?= pr_num($summary['avg_carga_min'] ?? 0, 1) ?> min · <strong>Promedio en planta:</strong> <?= pr_num($summary['avg_planta_min'] ?? 0, 1) ?> min</div>
  </div>

  <table>
    <thead>
      <tr>
        <th>#</th>
        <th>Código</th>
        <th>Fecha</th>
        <th>Depósito</th>
        <th>Estado</th>
        <th>Móvil</th>
        <th>Chofer</th>
        <th>Destinos</th>
        <th>Preembarques</th>
        <th>Espera</th>
        <th>Carga</th>
        <th>Planta</th>
        <th>Ayudantes</th>
        <th>Temp. salida</th>
        <th>Km inicial</th>
        <th>Observación</th>
      </tr>
    </thead>
    <tbody>
      <?php if (empty($rows)): ?>
        <tr>
          <td colspan="16" style="text-align:center; padding:16px;">Sin datos que coincidan con los filtros seleccionados.</td>
        </tr>
      <?php else: ?>
        <?php foreach ($rows as $index => $row): ?>
          <tr>
            <td class="num"><?= $index + 1 ?></td>
            <td><?= htmlspecialchars($row['codigo'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars(($row['fecha_creado'] ?? '-') . ($row['hora_creado'] ? ' ' . $row['hora_creado'] : ''), ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($row['deposito_nombre'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($row['estado_nombre'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($row['movil_chapa'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td><?= htmlspecialchars($row['chofer_nombre'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
            <td class="num"><?= pr_num($row['destinos'] ?? 0) ?></td>
            <td class="num"><?= pr_num($row['preembarques'] ?? 0) ?></td>
            <td class="num"><?= pr_num($row['tiempo_espera_min'] ?? 0) ?></td>
            <td class="num"><?= pr_num($row['tiempo_carga_min'] ?? 0) ?></td>
            <td class="num"><?= pr_num($row['tiempo_planta_min'] ?? 0) ?></td>
            <td class="num"><?= pr_num($row['ayudantes_cant'] ?? 0) ?></td>
            <td class="num"><?= pr_num($row['temp_salida_c'] ?? 0, 1) ?></td>
            <td class="num"><?= pr_num($row['km_inicial'] ?? 0, 1) ?></td>
            <td><?= htmlspecialchars($row['observacion'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
          </tr>
        <?php endforeach; ?>
      <?php endif; ?>
    </tbody>
    <tfoot>
      <tr>
        <td colspan="7">Totales</td>
        <td class="num"><?= pr_num($summary['total_destinos'] ?? 0) ?></td>
        <td class="num"><?= pr_num($summary['total_preembarques'] ?? 0) ?></td>
        <td class="num"><?= pr_num($summary['avg_espera_min'] ?? 0, 1) ?></td>
        <td class="num"><?= pr_num($summary['avg_carga_min'] ?? 0, 1) ?></td>
        <td class="num"><?= pr_num($summary['avg_planta_min'] ?? 0, 1) ?></td>
        <td class="num"><?= pr_num(array_sum(array_map(static fn($r) => (int) ($r['ayudantes_cant'] ?? 0), $rows))) ?></td>
        <td class="num"><?= pr_num(array_sum(array_map(static fn($r) => (float) ($r['temp_salida_c'] ?? 0), $rows)), 1) ?></td>
        <td class="num"><?= pr_num(array_sum(array_map(static fn($r) => (float) ($r['km_inicial'] ?? 0), $rows)), 1) ?></td>
        <td></td>
      </tr>
    </tfoot>
  </table>

  <?php if ($truncated): ?>
    <p class="warning" style="margin-top:10px;">
      Se muestran hasta <?= pr_num($limitUsed) ?> registros. Refine los filtros para obtener mayor detalle.
    </p>
  <?php endif; ?>
</body>
</html>
<?php
$html = ob_get_clean();

$mpdf = new \Mpdf\Mpdf([
    'mode' => 'utf-8',
    'format' => 'A4-L',
    'margin_left' => 10,
    'margin_right' => 10,
    'margin_top' => 12,
    'margin_bottom' => 12,
]);
$mpdf->SetTitle('Reporte de Salidas');
$mpdf->SetAuthor((string) env('APP_NAME', 'SOL'));
$mpdf->WriteHTML($html);
$mpdf->Output($filename, 'I');
exit;
