<?php

declare(strict_types=1);

if (!headers_sent()) {
    header('Content-Type: application/pdf; charset=utf-8');
}

$BASE = dirname(__DIR__, 3);
require_once $BASE . '/config/config.php';
require_once $BASE . '/config/db.php';
require_once $BASE . '/app/Support/Reportes/StockQuiebreReport.php';

if (empty($_SESSION['usuario_id'])) {
    http_response_code(401);
    echo 'No autorizado';
    exit;
}

$pdo = getPDO();
$pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

$limit = (int)($_GET['limit'] ?? 500);
$result = stock_quiebre_fetch_data($pdo, $_GET, $limit);
$combos = stock_quiebre_fetch_combos($pdo);

$filters = $result['filters'];

$clienteLabel = 'Todos';
if (!empty($filters['cliente_id'])) {
    $clienteLabel = 'Desconocido (' . $filters['cliente_id'] . ')';
    foreach ($combos['clientes'] as $cliente) {
        if ((string)$cliente['id'] === (string)$filters['cliente_id']) {
            $clienteLabel = $cliente['razon_social'];
            break;
        }
    }
}

$operativaLabel = 'Todas';
if (!empty($filters['operativa_id'])) {
    $operativaLabel = 'Desconocida (' . $filters['operativa_id'] . ')';
    foreach ($combos['operativas'] as $operativa) {
        if ((string)$operativa['id'] === (string)$filters['operativa_id']) {
            $operativaLabel = $operativa['nombre'];
            break;
        }
    }
}

$criterioLabel = $result['criterion'] === 'stock' ? 'Stock total' : 'Disponibles';
$soloMinLabel = !empty($filters['solo_con_min']) ? 'Sí' : 'No';
$qLabel = $filters['q'] !== '' ? $filters['q'] : 'Todos';

$generatedAt = (new DateTimeImmutable('now'))->format('d/m/Y H:i');
$filename = 'reporte_quiebre_stock_' . (new DateTimeImmutable('now'))->format('Ymd_His') . '.pdf';

function sqr_num($value, int $decimals = 0): string
{
    $num = (float)$value;
    return number_format($num, $decimals, ',', '.');
}

function sqr_percent(?float $value): string
{
    if ($value === null) {
        return '-';
    }
    return number_format($value, 1, ',', '.') . '%';
}

ob_start();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="utf-8">
    <title>Reporte de Quiebre de Stock</title>
    <style>
        body { font-family: 'DejaVu Sans', sans-serif; font-size: 10pt; color: #111; }
        h1 { font-size: 16pt; margin-bottom: 4px; }
        .meta { font-size: 9pt; margin-bottom: 16px; }
        .meta span { display: inline-block; margin-right: 12px; }
        table { width: 100%; border-collapse: collapse; margin-top: 12px; }
        th, td { border: 1px solid #bbb; padding: 6px 4px; }
        th { background: #f0f0f0; font-weight: bold; text-align: center; }
        td { text-align: left; }
        td.num { text-align: right; }
        tfoot td { font-weight: bold; background: #fafafa; }
        .badge { display: inline-block; padding: 2px 6px; border-radius: 4px; background: #efefef; font-size: 8pt; }
        .warning { color: #c0392b; }
    </style>
</head>
<body>
    <h1>Reporte de Quiebre de Stock</h1>
    <p class="meta">
        <span><strong>Generado:</strong> <?= htmlspecialchars($generatedAt, ENT_QUOTES, 'UTF-8') ?></span>
        <span><strong>Criterio:</strong> <?= htmlspecialchars($criterioLabel, ENT_QUOTES, 'UTF-8') ?></span>
        <span><strong>Cliente:</strong> <?= htmlspecialchars($clienteLabel, ENT_QUOTES, 'UTF-8') ?></span>
        <span><strong>Operativa:</strong> <?= htmlspecialchars($operativaLabel, ENT_QUOTES, 'UTF-8') ?></span>
        <span><strong>Producto:</strong> <?= htmlspecialchars($qLabel, ENT_QUOTES, 'UTF-8') ?></span>
        <span><strong>Solo con mínimo:</strong> <?= htmlspecialchars($soloMinLabel, ENT_QUOTES, 'UTF-8') ?></span>
        <span><strong>Registros:</strong> <?= sqr_num($result['totals']['productos'] ?? 0) ?></span>
    </p>

    <table>
        <thead>
            <tr>
                <th>#</th>
                <th>Cliente</th>
                <th>Operativa</th>
                <th>SKU</th>
                <th>Producto</th>
                <th>Mínimo</th>
                <th>Stock</th>
                <th>Reservados</th>
                <th>Disponibles</th>
                <th>Faltante</th>
                <th>Cobertura</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($result['rows'])): ?>
                <tr>
                    <td colspan="11" style="text-align:center; padding:18px;">Sin datos que coincidan con los filtros seleccionados.</td>
                </tr>
            <?php else: ?>
                <?php foreach ($result['rows'] as $index => $row): ?>
                <tr>
                    <td style="text-align:center;"><?= $index + 1 ?></td>
                    <td><?= htmlspecialchars($row['cliente'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= htmlspecialchars($row['operativa'] ?? '-', ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= htmlspecialchars($row['sku'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= htmlspecialchars($row['denominacion'] ?? '', ENT_QUOTES, 'UTF-8') ?></td>
                    <td class="num"><?= sqr_num($row['minimo'] ?? 0) ?></td>
                    <td class="num"><?= sqr_num($row['stock'] ?? 0) ?></td>
                    <td class="num"><?= sqr_num($row['reservados'] ?? 0) ?></td>
                    <td class="num"><?= sqr_num($row['disponibles'] ?? 0) ?></td>
                    <td class="num warning"><?= sqr_num($row['faltante_min'] ?? 0) ?></td>
                    <td class="num"><?= sqr_percent($row['cobertura_pct'] ?? null) ?></td>
                </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
        <tfoot>
            <tr>
                <td colspan="5">Totales</td>
                <td class="num"><?= sqr_num($result['totals']['minimo_total'] ?? 0) ?></td>
                <td class="num"><?= sqr_num($result['totals']['stock_total'] ?? 0) ?></td>
                <td class="num"><?= sqr_num($result['totals']['reservados_total'] ?? 0) ?></td>
                <td class="num"><?= sqr_num($result['totals']['disponibles_total'] ?? 0) ?></td>
                <td class="num warning"><?= sqr_num($result['totals']['faltante_total'] ?? 0) ?></td>
                <td class="num"><?= sqr_percent($result['totals']['cobertura_promedio'] ?? null) ?></td>
            </tr>
        </tfoot>
    </table>

    <?php if (!empty($result['truncated'])): ?>
        <p class="meta warning" style="margin-top:10px;">
            Se muestran hasta <?= sqr_num($result['limit']) ?> registros. Refine los filtros para obtener mayor detalle.
        </p>
    <?php endif; ?>
</body>
</html>
<?php
$html = ob_get_clean();

$mpdf = new \Mpdf\Mpdf([
    'mode' => 'utf-8',
    'format' => 'A4-L',
    'margin_left' => 10,
    'margin_right' => 10,
    'margin_top' => 12,
    'margin_bottom' => 12,
]);
$mpdf->SetTitle('Reporte de Quiebre de Stock');
$mpdf->SetAuthor((string)env('APP_NAME', 'SOL'));
$mpdf->WriteHTML($html);
$mpdf->Output($filename, 'I');
exit;
