<?php
declare(strict_types=1);

header('Content-Type: application/pdf');

$ROOT = dirname(__DIR__, 3);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

// ---- Intenta cargar mPDF ----
$aut = $ROOT . '/vendor/autoload.php';
if (!is_file($aut)) {
  http_response_code(500);
  echo "Falta vendor/autoload.php. Instala dependencias con Composer.";
  exit;
}
require_once $aut;

if (!class_exists(\Mpdf\Mpdf::class)) {
  http_response_code(500);
  echo "No está instalado mpdf/mpdf. Ejecuta: composer require mpdf/mpdf";
  exit;
}

// ---- Helpers mínimos ----
function nf($n, int $dec = 0): string {
  if ($n === null || $n === '') return '0';
  return number_format((float)$n, $dec, ',', '.'); // 1.234.567,89
}
function dateFormat($date, $format = 'Y-m-d H:i:s'): string {
  if (empty($date) || $date === '0000-00-00 00:00:00') return '-';
  try {
    return (new DateTime($date))->format($format);
  } catch (Exception $e) {
    return (string)$date;
  }
}

try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $pdo->exec('SET NAMES utf8mb4');

  $preId = isset($_GET['pre_id']) ? (int)$_GET['pre_id'] : 0;
  if ($preId <= 0) {
    http_response_code(400);
    echo "pre_id es requerido";
    exit;
  }

  // ---------- CABECERA DEL PREEMBARQUE ----------
  $sqlH = "SELECT pre.id, pre.codigo, pre.pedido_id, pre.asignado_at, pre.inicio_at, pre.fin_at,
                  ped.codigo AS pedido_codigo,
                  cli.razon_social AS cliente_nombre, cli.ruc AS cliente_codigo,
                  dep.nombre AS deposito_nombre,
                  est.nombre AS estado_nombre,
                  pos.code AS posicion_codigo
           FROM so_preembarque pre
           LEFT JOIN so_pedido ped ON ped.id = pre.pedido_id
           LEFT JOIN para_clientes cli ON cli.id = ped.cliente_id
           LEFT JOIN wh_deposito dep ON dep.id = pre.deposito_id
           LEFT JOIN so_preembarque_estado est ON est.id = pre.estado_id
           LEFT JOIN wh_posicion pos ON pos.id = pre.zona_posicion_id
           WHERE pre.id = ?
           LIMIT 1";
  $sth = $pdo->prepare($sqlH);
  $sth->execute([$preId]);
  $H = $sth->fetch(PDO::FETCH_ASSOC);

  if (!$H) {
    http_response_code(404);
    echo "Preembarque no encontrado";
    exit;
  }

  $pedidoId = (int)$H['pedido_id'];

  // ---------- MOVIMIENTOS ----------
  $sqlM = "SELECT m.id, m.created_at, m.tipo, m.motivo, m.producto_id, m.lote_id, m.pallet_id,
                  m.from_pos_id, m.to_pos_id, m.delta_uv, m.delta_uc, m.referencia,
                  p.sku AS producto_code, p.denominacion AS producto_nombre,
                  l.codigo AS lote_codigo, l.fecha_vencimiento,
                  pf.code AS from_code, pt.code AS to_code,
                  pal.codigo AS pallet_codigo
           FROM wh_move m
           LEFT JOIN para_productos p ON p.id=m.producto_id
           LEFT JOIN wh_lote l ON l.id=m.lote_id
           LEFT JOIN wh_posicion pf ON pf.id=m.from_pos_id
           LEFT JOIN wh_posicion pt ON pt.id=m.to_pos_id
           LEFT JOIN wh_pallet pal ON pal.id=m.pallet_id
           WHERE m.referencia IN (CONCAT('PRE-', ?), CONCAT('PREP-REP-', ?))
             AND m.motivo IN ('PREPARACION', 'REPOSICION_PICKING')
           ORDER BY m.created_at ASC, m.id ASC";
  $stm = $pdo->prepare($sqlM);
  $stm->execute([$pedidoId, $pedidoId]);
  $moves = $stm->fetchAll(PDO::FETCH_ASSOC);

  // ---------- ESTADÍSTICAS ----------
  $totalMovs = count($moves);
  $totalUV = 0;
  $totalUC = 0;
  $porMotivo = [];
  $porTipo = [];

  foreach ($moves as $mv) {
    $totalUV += abs((float)$mv['delta_uv']);
    $totalUC += abs((float)$mv['delta_uc']);
    
    $motivo = $mv['motivo'] ?? 'Sin motivo';
    $porMotivo[$motivo] = ($porMotivo[$motivo] ?? 0) + 1;
    
    $tipo = $mv['tipo'] ?? 'Sin tipo';
    $porTipo[$tipo] = ($porTipo[$tipo] ?? 0) + 1;
  }

  // ---------- ÍTEMS DEL PEDIDO ----------
  $sqlI = "SELECT i.producto_id, i.expected_uv, i.expected_uc, i.prepared_uv, i.prepared_uc,
                  p.sku, p.denominacion,
                  (i.expected_uv - COALESCE(i.prepared_uv, 0)) AS pending_uv,
                  (i.expected_uc - COALESCE(i.prepared_uc, 0)) AS pending_uc
           FROM so_pedido_dest_item i
           LEFT JOIN para_productos p ON p.id = i.producto_id
           LEFT JOIN so_pedido_dest pd ON pd.id = i.pedido_dest_id
           WHERE pd.pedido_id = ?
           ORDER BY p.sku ASC";
  $sti = $pdo->prepare($sqlI);
  $sti->execute([$pedidoId]);
  $items = $sti->fetchAll(PDO::FETCH_ASSOC);

  // ---------- RENDER PDF ----------
  $mpdf = new \Mpdf\Mpdf([
    'format' => 'A4',
    'margin_left'   => 10,
    'margin_right'  => 10,
    'margin_top'    => 14,
    'margin_bottom' => 12,
    'orientation'   => 'P'
  ]);

  $title = "Movimientos de Preparación - {$H['codigo']}";
  $styles = '
    <style>
      body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 10px; }
      h1 { font-size: 16px; margin: 0 0 8px 0; color: #2c3e50; }
      h2 { font-size: 13px; margin: 12px 0 6px 0; color: #34495e; }
      .muted { color:#666; }
      .grid { width:100%; border-collapse: collapse; margin-top:6px; }
      .grid th, .grid td { border: 0.5px solid #bdc3c7; padding: 6px 8px; }
      .grid th { background: #ecf0f1; font-weight: bold; }
      .grid tbody td:nth-child(odd) { background: #f8f9fa; font-weight: bold; }
      .grid tbody td:nth-child(even) { background: #ffffff; }
      .right { text-align:right; }
      .center { text-align:center; }
      .small { font-size: 9px; }
      .section { margin-top: 12px; }
      .kv { display:flex; flex-wrap:wrap; gap:15px; margin-bottom: 8px; }
      .kv div { min-width: 180px; }
      .two-column { display: flex; justify-content: space-between; }
      .column { width: 48%; }
      .column div { margin-bottom: 6px; }
      .stats { background: #f8f9fa; padding: 8px; border-radius: 4px; margin: 8px 0; }
      .badge { 
        display:inline-block; 
        padding:2px 6px; 
        background:transparent; 
        color:#333; 
        border:1px solid #ddd;
        border-radius:3px; 
        font-size: 9px;
        font-weight: bold;
      }
      .badge.success { background: transparent; color: #333; border-color: #ddd; }
      .badge.warning { background: transparent; color: #333; border-color: #ddd; }
      .badge.danger { background: transparent; color: #333; border-color: #ddd; }
    </style>
  ';

  ob_start();
  ?>
  <?= $styles ?>
  <h1><?= htmlspecialchars($title) ?></h1>
  <div class="muted small">Generado: <?= date('Y-m-d H:i:s') ?></div>

  <div class="section">
    <h2>Información del Preembarque</h2>
    <table class="grid">
      <tbody>
        <tr>
          <td><b>Código:</b></td>
          <td><?= htmlspecialchars((string)$H['codigo']) ?></td>
          <td><b>Pedido:</b></td>
          <td><?= htmlspecialchars((string)$H['pedido_codigo']) ?></td>
        </tr>
        <tr>
          <td><b>Cliente:</b></td>
          <td><?= htmlspecialchars((string)($H['cliente_nombre'] ?? $H['cliente_codigo'] ?? '-')) ?></td>
          <td><b>Depósito:</b></td>
          <td><?= htmlspecialchars((string)($H['deposito_nombre'] ?? '-')) ?></td>
        </tr>
        <tr>
          <td><b>Estado:</b></td>
          <td><?= htmlspecialchars((string)($H['estado_nombre'] ?? '-')) ?></td>
          <td><b>Posición:</b></td>
          <td><?= htmlspecialchars((string)($H['posicion_codigo'] ?? '-')) ?></td>
        </tr>
        <tr>
          <td><b>Asignado:</b></td>
          <td><?= dateFormat($H['asignado_at']) ?></td>
          <td><b>Iniciado:</b></td>
          <td><?= dateFormat($H['inicio_at']) ?></td>
        </tr>
        <tr>
          <td><b>Finalizado:</b></td>
          <td><?= dateFormat($H['fin_at']) ?></td>
          <td></td>
          <td></td>
        </tr>
      </tbody>
    </table>
  </div>

  <div class="section stats">
    <h2>Estadísticas de Movimientos</h2>
    <table class="grid">
      <tbody>
        <tr>
          <td><b>Total movimientos:</b></td>
          <td><span class="badge"><?= $totalMovs ?></span></td>
          <td><b>Total UV movidas:</b></td>
          <td><span class="badge success"><?= nf($totalUV) ?></span></td>
        </tr>
        <tr>
          <td><b>Total UC movidas:</b></td>
          <td><span class="badge success"><?= nf($totalUC) ?></span></td>
          <?php if (!empty($porMotivo)): ?>
          <td><b>Por motivo:</b></td>
          <td>
            <?php foreach ($porMotivo as $motivo => $count): ?>
              <span class="badge warning"><?= htmlspecialchars($motivo) ?>: <?= $count ?></span>
            <?php endforeach; ?>
          </td>
          <?php else: ?>
          <td></td>
          <td></td>
          <?php endif; ?>
        </tr>
        <?php if (!empty($porTipo)): ?>
        <tr>
          <td><b>Por tipo:</b></td>
          <td colspan="3">
            <?php foreach ($porTipo as $tipo => $count): ?>
              <span class="badge"><?= htmlspecialchars($tipo) ?>: <?= $count ?></span>
            <?php endforeach; ?>
          </td>
        </tr>
        <?php endif; ?>
      </tbody>
    </table>
  </div>

  <div class="section">
    <h2>Ítems del Pedido - Totales a Embarcar</h2>
    <table class="grid">
      <thead>
        <tr>
          <th>SKU</th>
          <th>Producto</th>
          <th class="right">A Embarcar UV</th>
          <th class="right">A Embarcar UC</th>
        </tr>
      </thead>
      <tbody>
      <?php foreach ($items as $it): ?>
        <tr>
          <td><?= htmlspecialchars((string)($it['sku'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($it['denominacion'] ?? '')) ?></td>
          <td class="right"><?= nf($it['expected_uv']) ?></td>
          <td class="right"><?= nf($it['expected_uc']) ?></td>
        </tr>
      <?php endforeach; ?>
      </tbody>
    </table>
  </div>

  <div class="section">
    <h2>Detalle de Movimientos</h2>
    <table class="grid">
      <thead>
        <tr>
          <th>Motivo</th>
          <th>SKU</th>
          <th>Producto</th>
          <th>Lote</th>
          <th>Vencimiento</th>
          <th>Pallet</th>
          <th>Desde</th>
          <th>Hacia</th>
          <th class="right">UV</th>
          <th class="right">UC</th>
        </tr>
      </thead>
      <tbody>
      <?php foreach ($moves as $mv): ?>
        <tr>
          <td class="small"><?= htmlspecialchars((string)($mv['motivo'] ?? '')) ?></td>
          <td><?= htmlspecialchars((string)($mv['producto_code'] ?? '')) ?></td>
          <td class="small"><?= htmlspecialchars((string)($mv['producto_nombre'] ?? '')) ?></td>
          <td class="small"><?= htmlspecialchars((string)($mv['lote_codigo'] ?? '')) ?></td>
          <td class="small"><?= dateFormat($mv['fecha_vencimiento'], 'd/m/Y') ?></td>
          <td class="small"><?= htmlspecialchars((string)($mv['pallet_codigo'] ?? '')) ?></td>
          <td class="small"><?= htmlspecialchars((string)($mv['from_code'] ?? '')) ?></td>
          <td class="small"><?= htmlspecialchars((string)($mv['to_code'] ?? '')) ?></td>
          <td class="right"><?= nf(abs((float)$mv['delta_uv'])) ?></td>
          <td class="right"><?= nf(abs((float)$mv['delta_uc'])) ?></td>
        </tr>
      <?php endforeach; ?>
      </tbody>
      <tfoot>
        <tr>
          <th colspan="8" class="right">Totales</th>
          <th class="right"><?= nf($totalUV) ?></th>
          <th class="right"><?= nf($totalUC) ?></th>
        </tr>
      </tfoot>
    </table>
  </div>

  <?php if (empty($moves)): ?>
  <div class="section center">
    <div style="padding: 20px; background: #f8f9fa; border-radius: 4px;">
      <span class="badge warning">Sin movimientos</span>
      <p class="muted">No se encontraron movimientos de preparación para este preembarque.</p>
    </div>
  </div>
  <?php endif; ?>

  <div class="section">
    <h2>Para Completar Manualmente</h2>
    <div style="background: #f8f9fa; padding: 15px; border-radius: 4px; margin-top: 15px;">
      <div style="display: flex; justify-content: space-between; margin-bottom: 20px;">
        <div style="width: 48%;">
          <div style="margin-bottom: 15px;">
            <b>Montacarguista:</b>
            <div style="border-bottom: 1px solid #333; height: 20px; margin-top: 5px;"></div>
          </div>
          <div style="margin-bottom: 15px;">
            <b>Fecha/Hora Inicio:</b>
            <div style="border-bottom: 1px solid #333; height: 20px; margin-top: 5px;"></div>
          </div>
        </div>
        <div style="width: 48%;">
          <div style="margin-bottom: 15px;">
            <b>Fecha/Hora Fin:</b>
            <div style="border-bottom: 1px solid #333; height: 20px; margin-top: 5px;"></div>
          </div>
        </div>
      </div>
      <div>
        <b>Observaciones:</b>
        <div style="border-bottom: 1px solid #333; height: 20px; margin-top: 5px;"></div>
        <div style="border-bottom: 1px solid #333; height: 20px; margin-top: 10px;"></div>
        <div style="border-bottom: 1px solid #333; height: 20px; margin-top: 10px;"></div>
      </div>
    </div>
  </div>
  
  <?php
  $html = ob_get_clean();

  $mpdf->SetTitle($title);
  $mpdf->SetAuthor((string)env('APP_NAME','SOL'));
  $mpdf->WriteHTML($html);
  $mpdf->Output("movimientos_prep_{$H['codigo']}.pdf", \Mpdf\Output\Destination::INLINE);

} catch (Throwable $e) {
  http_response_code(500);
  header('Content-Type: text/plain; charset=utf-8');
  echo (env('APP_ENV')==='local') ? ("Error generando PDF: ".$e->getMessage()) : "No se pudo generar el PDF";
}