<?php
declare(strict_types=1);

/**
 * PDF: Hoja de Reposición para Montacarguista
 * Muestra solo movimientos de reposición (REPOSICION_PICKING)
 */

header('Content-Type: application/pdf');

$ROOT = dirname(__DIR__, 3);
require_once $ROOT . '/config/config.php';
require_once $ROOT . '/config/db.php';

// ---- Intenta cargar mPDF ----
$aut = $ROOT . '/vendor/autoload.php';
if (!is_file($aut)) {
  http_response_code(500);
  echo "Falta vendor/autoload.php. Instala dependencias con Composer.";
  exit;
}
require_once $aut;

if (!class_exists(\Mpdf\Mpdf::class)) {
  http_response_code(500);
  echo "No está instalado mpdf/mpdf. Ejecuta: composer require mpdf/mpdf";
  exit;
}

// ---- Helpers mínimos ----
function nf($n, int $dec = 0): string {
  if ($n === null || $n === '') return '0';
  return number_format((float)$n, $dec, ',', '.'); // 1.234.567,89
}
function dateFormat($date, $format = 'Y-m-d H:i:s'): string {
  if (empty($date) || $date === '0000-00-00 00:00:00') return '-';
  try {
    return (new DateTime($date))->format($format);
  } catch (Exception $e) {
    return (string)$date;
  }
}

try {
  $pdo = get_pdo();
  $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
  $pdo->exec('SET NAMES utf8mb4');

  $preId = isset($_GET['pre_id']) ? (int)$_GET['pre_id'] : 0;
  $soId  = isset($_GET['so_id']) ? (int)$_GET['so_id'] : 0;
  if ($preId <= 0 && $soId <= 0) {
    http_response_code(400);
    echo "pre_id o so_id es requerido";
    exit;
  }

  // ---------- CABECERA DEL PREEMBARQUE ----------
  $H = null;
  $pedidoId = 0;
  if ($preId > 0) {
    $sqlH = "SELECT pre.id, pre.codigo, pre.pedido_id, pre.asignado_at, pre.inicio_at, pre.fin_at,
                    ped.codigo AS pedido_codigo,
                    cli.razon_social AS cliente_nombre, cli.ruc AS cliente_codigo,
                    dep.nombre AS deposito_nombre,
                    est.nombre AS estado_nombre,
                    pos.code AS posicion_codigo
             FROM so_preembarque pre
             LEFT JOIN so_pedido ped ON ped.id = pre.pedido_id
             LEFT JOIN para_clientes cli ON cli.id = ped.cliente_id
             LEFT JOIN wh_deposito dep ON dep.id = pre.deposito_id
             LEFT JOIN so_preembarque_estado est ON est.id = pre.estado_id
             LEFT JOIN wh_posicion pos ON pos.id = pre.zona_posicion_id
             WHERE pre.id = ?
             LIMIT 1";
    $sth = $pdo->prepare($sqlH);
    $sth->execute([$preId]);
    $H = $sth->fetch(PDO::FETCH_ASSOC);
    if (!$H) {
      http_response_code(404);
      echo "Preembarque no encontrado";
      exit;
    }
    $pedidoId = (int)$H['pedido_id'];
  } else {
    // Fallback por so_id (sin preembarque): armar encabezado mínimo con datos del pedido
    $sth = $pdo->prepare("SELECT p.id, p.codigo, p.cliente_id, cli.razon_social AS cliente_nombre, cli.ruc AS cliente_codigo
                           FROM so_pedido p
                           LEFT JOIN para_clientes cli ON cli.id = p.cliente_id
                           WHERE p.id = ? LIMIT 1");
    $sth->execute([$soId]);
    $P = $sth->fetch(PDO::FETCH_ASSOC);
    if (!$P) {
      http_response_code(404);
      echo "Pedido no encontrado";
      exit;
    }
    $pedidoId = (int)$P['id'];
    $H = [
      'id' => null,
      'codigo' => (string)$P['codigo'],
      'pedido_id' => (int)$P['id'],
      'pedido_codigo' => (string)$P['codigo'],
      'cliente_nombre' => $P['cliente_nombre'] ?? null,
      'cliente_codigo' => $P['cliente_codigo'] ?? null,
      'deposito_nombre' => null,
      'estado_nombre' => 'SIN PREEMBARQUE',
      'posicion_codigo' => null,
      'asignado_at' => null,
      'inicio_at' => null,
      'fin_at' => null,
    ];
  }

  // ---------- MOVIMIENTOS DE REPOSICIÓN ÚNICAMENTE ----------
  $sqlM = "SELECT m.id, m.created_at, m.tipo, m.motivo, m.producto_id, m.lote_id, m.pallet_id,
                  m.from_pos_id, m.to_pos_id, m.delta_uv, m.delta_uc, m.referencia,
                  p.sku AS producto_code, p.denominacion AS producto_nombre,
                  l.codigo AS lote_codigo, l.fecha_vencimiento,
                  pf.code AS from_code, pt.code AS to_code,
                  pal.codigo AS pallet_codigo
           FROM wh_move m
           LEFT JOIN para_productos p ON p.id=m.producto_id
           LEFT JOIN wh_lote l ON l.id=m.lote_id
           LEFT JOIN wh_posicion pf ON pf.id=m.from_pos_id
           LEFT JOIN wh_posicion pt ON pt.id=m.to_pos_id
           LEFT JOIN wh_pallet pal ON pal.id=m.pallet_id
           WHERE m.referencia = CONCAT('PREP-REP-', ?)
             AND m.motivo = 'REPOSICION_PICKING'
           ORDER BY m.created_at ASC, m.id ASC";
  $moves = [];
  if ($preId > 0) {
    $stm = $pdo->prepare($sqlM);
    $stm->execute([$pedidoId]);
    $moves = $stm->fetchAll(PDO::FETCH_ASSOC);
  }

  // ---------- FALLBACK: Sugerir reposición si no hay movimientos cargados ----------
  if (empty($moves)) {
    // 1) Agregar requerimientos por producto desde el pedido
    $reqSql = "
      SELECT i.producto_id,
             pr.sku AS producto_code,
             pr.denominacion AS producto_nombre,
             SUM(i.expected_uv) AS need_uv,
             SUM(i.expected_uc) AS need_uc
        FROM so_pedido_dest d
        JOIN so_pedido_dest_item i ON i.pedido_dest_id = d.id
        JOIN para_productos pr ON pr.id = i.producto_id
       WHERE d.pedido_id = ?
       GROUP BY i.producto_id, pr.sku, pr.denominacion
    ";
    $stReq = $pdo->prepare($reqSql);
    $stReq->execute([$pedidoId]);
    $reqs = $stReq->fetchAll(PDO::FETCH_ASSOC) ?: [];

    // 2) Determinar un destino "pick" representativo para mostrar (código)
    $pickPosCode = '-';
    $pickPosId = null;
    try {
      if ($preId > 0) {
        $stPick = $pdo->prepare("SELECT p.id, p.code
                                   FROM wh_posicion p
                                   JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PICKING'
                                  WHERE p.deposito_id=(SELECT deposito_id FROM so_preembarque WHERE id=? LIMIT 1)
                                  ORDER BY p.id LIMIT 1");
        $stPick->execute([$preId]);
      } else {
        $stPick = $pdo->prepare("SELECT p.id, p.code
                                   FROM wh_posicion p
                                   JOIN wh_ambiente a ON a.id=p.ambiente_id AND a.code='PICKING'
                                  ORDER BY p.id LIMIT 1");
        $stPick->execute();
      }
      $pRow = $stPick->fetch(PDO::FETCH_ASSOC);
      if ($pRow) { $pickPosId = (int)$pRow['id']; $pickPosCode = (string)$pRow['code']; }
    } catch (Throwable $e) { /* ignore */ }

    // 3) Para cada producto, calcular déficit en zona PICK (wh_posicion.is_pick_face=1)
    $moves = [];
    foreach ($reqs as $rq) {
      $prodId = (int)$rq['producto_id'];
      $needUv = (int)($rq['need_uv'] ?? 0);
      $needUc = (int)($rq['need_uc'] ?? 0);

      // Stock actual en pick-face (UV/UC)
      if ($preId > 0) {
        $stPickStock = $pdo->prepare("\n          SELECT COALESCE(SUM(s.qty_uv),0) AS uv, COALESCE(SUM(s.qty_uc),0) AS uc\n            FROM wh_stock s\n            JOIN wh_posicion pos ON pos.id = s.posicion_id\n            JOIN wh_ambiente a ON a.id=pos.ambiente_id AND a.code='PICKING'\n           WHERE s.producto_id=?\n             AND s.deposito_id = (SELECT deposito_id FROM so_preembarque WHERE id=? LIMIT 1)\n        ");
        $stPickStock->execute([$prodId, $preId]);
      } else {
        $stPickStock = $pdo->prepare("\n          SELECT COALESCE(SUM(s.qty_uv),0) AS uv, COALESCE(SUM(s.qty_uc),0) AS uc\n            FROM wh_stock s\n            JOIN wh_posicion pos ON pos.id = s.posicion_id\n            JOIN wh_ambiente a ON a.id=pos.ambiente_id AND a.code='PICKING'\n           WHERE s.producto_id=?\n        ");
        $stPickStock->execute([$prodId]);
      }
      $ps = $stPickStock->fetch(PDO::FETCH_ASSOC) ?: ['uv'=>0,'uc'=>0];
      $haveUv = (int)$ps['uv'];
      $haveUc = (int)$ps['uc'];

      $defUv = max($needUv - $haveUv, 0);
      $defUc = max($needUc - $haveUc, 0);
      if ($defUv <= 0 && $defUc <= 0) continue; // no hace falta reposición para este producto

      // 4) Buscar fuentes en reserva (posiciones que no son pick-face) por FEFO (vencimiento más próximo)
      if ($preId > 0) {
        $stSrc = $pdo->prepare("\n          SELECT s.posicion_id, pf.code AS from_code, s.pallet_id, pal.codigo AS pallet_codigo,\n                 s.lote_id, l.codigo AS lote_codigo, l.fecha_vencimiento,\n                 s.qty_uv, s.qty_uc\n            FROM wh_stock s\n            JOIN wh_posicion pf ON pf.id = s.posicion_id\n            JOIN wh_ambiente a ON a.id=pf.ambiente_id AND a.code NOT IN ('PREP','CUARENTENA','PICKING')\n            LEFT JOIN wh_lote l ON l.id = s.lote_id\n            LEFT JOIN wh_pallet pal ON pal.id = s.pallet_id\n           WHERE s.deposito_id = (SELECT deposito_id FROM so_preembarque WHERE id=? LIMIT 1)\n             AND s.producto_id = ?\n             AND (s.qty_uv > 0 OR s.qty_uc > 0)\n           ORDER BY ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC, s.posicion_id ASC\n        ");
        $stSrc->execute([$preId, $prodId]);
      } else {
        $stSrc = $pdo->prepare("\n          SELECT s.posicion_id, pf.code AS from_code, s.pallet_id, pal.codigo AS pallet_codigo,\n                 s.lote_id, l.codigo AS lote_codigo, l.fecha_vencimiento,\n                 s.qty_uv, s.qty_uc\n            FROM wh_stock s\n            JOIN wh_posicion pf ON pf.id = s.posicion_id\n            JOIN wh_ambiente a ON a.id=pf.ambiente_id AND a.code NOT IN ('PREP','CUARENTENA','PICKING')\n            LEFT JOIN wh_lote l ON l.id = s.lote_id\n            LEFT JOIN wh_pallet pal ON pal.id = s.pallet_id\n           WHERE s.producto_id = ?\n             AND (s.qty_uv > 0 OR s.qty_uc > 0)\n           ORDER BY ISNULL(l.fecha_vencimiento) ASC, l.fecha_vencimiento ASC, s.posicion_id ASC\n        ");
        $stSrc->execute([$prodId]);
      }
      $srcs = $stSrc->fetchAll(PDO::FETCH_ASSOC) ?: [];

      $remainUv = $defUv;
      $remainUc = $defUc;

      foreach ($srcs as $srow) {
        if ($remainUv <= 0 && $remainUc <= 0) break;
        $moveUv = min((int)$srow['qty_uv'], $remainUv);
        $moveUc = $moveUv > 0 ? 0 : min((int)$srow['qty_uc'], $remainUc); // priorizar UV
        if ($moveUv <= 0 && $moveUc <= 0) continue;

        $moves[] = [
          'producto_code' => (string)$rq['producto_code'],
          'producto_nombre' => (string)$rq['producto_nombre'],
          'lote_codigo' => (string)($srow['lote_codigo'] ?? ''),
          'fecha_vencimiento' => $srow['fecha_vencimiento'] ?? null,
          'pallet_codigo' => (string)($srow['pallet_codigo'] ?? ''),
          'from_code' => (string)$srow['from_code'],
          'to_code' => $pickPosCode ?: 'ZONA PICKING',
          'delta_uv' => $moveUv,
          'delta_uc' => $moveUc,
        ];

        $remainUv -= $moveUv;
        $remainUc -= $moveUc;
      }
    }
  }

  // ---------- ESTADÍSTICAS ----------
  $totalMovs = count($moves);
  $totalUV = 0;
  $totalUC = 0;

  foreach ($moves as $mv) {
    $totalUV += abs((float)$mv['delta_uv']);
    $totalUC += abs((float)$mv['delta_uc']);
  }

  // ---------- RENDER PDF ----------
  $mpdf = new \Mpdf\Mpdf([
    'format' => 'A4',
    'margin_left'   => 10,
    'margin_right'  => 10,
    'margin_top'    => 14,
    'margin_bottom' => 12,
    'orientation'   => 'P'
  ]);

  $title = "Hoja de Reposición - {$H['codigo']}";
  $styles = '
    <style>
      body { font-family: DejaVu Sans, Arial, sans-serif; font-size: 11px; }
      h1 { font-size: 18px; margin: 0 0 10px 0; color: #2c3e50; text-align: center; }
      h2 { font-size: 14px; margin: 15px 0 8px 0; color: #34495e; }
      .muted { color:#666; }
      .grid { width:100%; border-collapse: collapse; margin-top:8px; }
      .grid th, .grid td { border: 0.5px solid #bdc3c7; padding: 8px 6px; }
      .grid th { background: #ecf0f1; font-weight: bold; }
      .grid tbody td:nth-child(odd) { background: #f8f9fa; font-weight: bold; }
      .grid tbody td:nth-child(even) { background: #ffffff; }
      .right { text-align:right; }
      .center { text-align:center; }
      .small { font-size: 10px; }
      .section { margin-top: 15px; }
      .stats { background: #f8f9fa; padding: 10px; border-radius: 4px; margin: 10px 0; }
      .badge { 
        display:inline-block; 
        padding:3px 8px; 
        background:transparent; 
        color:#333; 
        border:1px solid #ddd;
        border-radius:4px; 
        font-size: 10px;
        font-weight: bold;
      }
      .warning-box {
        background: #fff3cd;
        border: 2px solid #ffc107;
        padding: 15px;
        border-radius: 6px;
        margin: 15px 0;
        text-align: center;
        font-weight: bold;
        color: #856404;
      }
      .footer { 
        margin-top: 30px; 
        padding-top: 15px; 
        border-top: 1px solid #dee2e6;
        font-size: 10px;
        color: #6c757d;
        text-align: center;
      }
    </style>';

  ob_start();
  ?>
  <?= $styles ?>
  <h1><?= htmlspecialchars($title) ?></h1>
  <div class="muted small center">Generado: <?= date('Y-m-d H:i:s') ?></div>

  <div class="warning-box">
    🚚 DOCUMENTO PARA MONTACARGUISTA - MOVIMIENTOS DE REPOSICIÓN 🚚
  </div>

  <!-- Información interna del preembarque removida para el documento de campo -->

  <div class="section stats">
    <h2>Resumen de Reposición</h2>
    <table class="grid">
      <tbody>
        <tr>
          <td><b>Total movimientos:</b></td>
          <td><span class="badge"><?= $totalMovs ?></span></td>
          <td><b>Total UV a mover:</b></td>
          <td><span class="badge"><?= nf($totalUV) ?></span></td>
        </tr>
        <tr>
          <td><b>Total UC a mover:</b></td>
          <td><span class="badge"><?= nf($totalUC) ?></span></td>
          <td><b>Objetivo:</b></td>
          <td>Reponer PICKING según FEFO</td>
        </tr>
      </tbody>
    </table>
  </div>

  <?php if (empty($moves)): ?>
  <div class="section center">
    <div style="padding: 20px; background: #f8f9fa; border-radius: 4px;">
      <span class="badge">Sin movimientos de reposición</span>
      <p class="muted">No se requieren movimientos de reposición para este pedido.</p>
    </div>
  </div>
  <?php else: ?>

  <div class="section">
    <h2>Movimientos de Reposición a Realizar</h2>
    <table class="grid">
      <thead>
        <tr>
          <th>SKU</th>
          <th>Producto</th>
          <th>Lote</th>
          <th>Vencimiento</th>
          <th>Pallet</th>
          <th>Desde</th>
          <th>Hacia</th>
          <th class="right">UV</th>
          <th class="right">UC</th>
          <th>✓</th>
        </tr>
      </thead>
      <tbody>
      <?php foreach ($moves as $mv): ?>
        <tr>
          <td class="small"><?= htmlspecialchars((string)($mv['producto_code'] ?? '')) ?></td>
          <td class="small"><?= htmlspecialchars((string)($mv['producto_nombre'] ?? '')) ?></td>
          <td class="small"><?= htmlspecialchars((string)($mv['lote_codigo'] ?? '')) ?></td>
          <td class="small"><?= dateFormat($mv['fecha_vencimiento'], 'd/m/Y') ?></td>
          <td class="small"><?= htmlspecialchars((string)($mv['pallet_codigo'] ?? '')) ?></td>
          <td class="small"><b><?= htmlspecialchars((string)($mv['from_code'] ?? '')) ?></b></td>
          <td class="small"><b><?= htmlspecialchars((string)($mv['to_code'] ?? '')) ?></b></td>
          <td class="right"><?= nf(abs((float)$mv['delta_uv'])) ?></td>
          <td class="right"><?= nf(abs((float)$mv['delta_uc'])) ?></td>
          <td class="center">☐</td>
        </tr>
      <?php endforeach; ?>
      </tbody>
      <tfoot>
        <tr>
          <th colspan="7" class="right">Totales</th>
          <th class="right"><?= nf($totalUV) ?></th>
          <th class="right"><?= nf($totalUC) ?></th>
          <th></th>
        </tr>
      </tfoot>
    </table>
  </div>

  <div class="section">
    <h2>Para Completar Manualmente</h2>
    <div style="background: #f8f9fa; padding: 15px; border-radius: 4px; margin-top: 15px;">
      <div style="display: flex; justify-content: space-between; margin-bottom: 20px;">
        <div style="width: 48%;">
          <div style="margin-bottom: 15px;">
            <b>Montacarguista:</b>
            <div style="border-bottom: 1px solid #333; height: 20px; margin-top: 5px;"></div>
          </div>
          <div style="margin-bottom: 15px;">
            <b>Fecha/Hora Inicio:</b>
            <div style="border-bottom: 1px solid #333; height: 20px; margin-top: 5px;"></div>
          </div>
        </div>
        <div style="width: 48%;">
          <div style="margin-bottom: 15px;">
            <b>Fecha/Hora Fin:</b>
            <div style="border-bottom: 1px solid #333; height: 20px; margin-top: 5px;"></div>
          </div>
        </div>
      </div>
      <div>
        <b>Observaciones:</b>
        <div style="border-bottom: 1px solid #333; height: 20px; margin-top: 5px;"></div>
        <div style="border-bottom: 1px solid #333; height: 20px; margin-top: 10px;"></div>
      </div>
    </div>
  </div>

  <?php endif; ?>

  <div class="footer">
    Generado el <?= date('d/m/Y H:i:s') ?> | Sistema SOL - Hoja de Reposición
  </div>
  
  <?php
  $html = ob_get_clean();

  $mpdf->SetTitle($title);
  $mpdf->SetAuthor((string)env('APP_NAME','SOL'));
  $mpdf->WriteHTML($html);
  $mpdf->Output("reposicion_{$H['codigo']}.pdf", \Mpdf\Output\Destination::INLINE);

} catch (Throwable $e) {
  http_response_code(500);
  header('Content-Type: text/plain; charset=utf-8');
  echo (env('APP_ENV')==='local') ? ("Error generando PDF: ".$e->getMessage()) : "No se pudo generar el PDF";
}
?>